/********************************************************************
*
* Standard register and bit definitions for the Texas Instruments
* MSP430 microcontroller.
*
* This file supports assembler and C development for
* MSP430x22x4 devices.
*
* Texas Instruments, Version 1.3
*
* Rev. 1.0, Setup
* Rev. 1.1, Updated names for USCI
* Rev. 1.2, Added missing definitions in OA and ADC10
* Rev. 1.3  Removed bit definitions for ADC10AEx as this are in conflict with ADC10AEx Register
*           Removed not existing SWCTL register definition
*
********************************************************************/

#ifndef __msp430x22x4
#define __msp430x22x4
               

/************************************************************
* STANDARD BITS
************************************************************/

#define BIT0                (0x0001)
#define BIT1                (0x0002)
#define BIT2                (0x0004)
#define BIT3                (0x0008)
#define BIT4                (0x0010)
#define BIT5                (0x0020)
#define BIT6                (0x0040)
#define BIT7                (0x0080)
#define BIT8                (0x0100)
#define BIT9                (0x0200)
#define BITA                (0x0400)
#define BITB                (0x0800)
#define BITC                (0x1000)
#define BITD                (0x2000)
#define BITE                (0x4000)
#define BITF                (0x8000)

/************************************************************
* STATUS REGISTER BITS
************************************************************/

#define C                   (0x0001)
#define Z                   (0x0002)
#define N                   (0x0004)
#define V                   (0x0100)
#define GIE                 (0x0008)
#define CPUOFF              (0x0010)
#define OSCOFF              (0x0020)
#define SCG0                (0x0040)
#define SCG1                (0x0080)

/* Low Power Modes coded with Bits 4-7 in SR */

#ifndef __STDC__ /* Begin #defines for assembler */
#define LPM0                (CPUOFF)
#define LPM1                (SCG0+CPUOFF)
#define LPM2                (SCG1+CPUOFF)
#define LPM3                (SCG1+SCG0+CPUOFF)
#define LPM4                (SCG1+SCG0+OSCOFF+CPUOFF)
/* End #defines for assembler */

#else /* Begin #defines for C */
#define LPM0_bits           (CPUOFF)
#define LPM1_bits           (SCG0+CPUOFF)
#define LPM2_bits           (SCG1+CPUOFF)
#define LPM3_bits           (SCG1+SCG0+CPUOFF)
#define LPM4_bits           (SCG1+SCG0+OSCOFF+CPUOFF)

#include <In430.h>

#define LPM0      _BIS_SR(LPM0_bits)     /* Enter Low Power Mode 0 */
#define LPM0_EXIT _BIC_SR_IRQ(LPM0_bits) /* Exit Low Power Mode 0 */
#define LPM1      _BIS_SR(LPM1_bits)     /* Enter Low Power Mode 1 */
#define LPM1_EXIT _BIC_SR_IRQ(LPM1_bits) /* Exit Low Power Mode 1 */
#define LPM2      _BIS_SR(LPM2_bits)     /* Enter Low Power Mode 2 */
#define LPM2_EXIT _BIC_SR_IRQ(LPM2_bits) /* Exit Low Power Mode 2 */
#define LPM3      _BIS_SR(LPM3_bits)     /* Enter Low Power Mode 3 */
#define LPM3_EXIT _BIC_SR_IRQ(LPM3_bits) /* Exit Low Power Mode 3 */
#define LPM4      _BIS_SR(LPM4_bits)     /* Enter Low Power Mode 4 */
#define LPM4_EXIT _BIC_SR_IRQ(LPM4_bits) /* Exit Low Power Mode 4 */
#endif /* End #defines for C */

/************************************************************
* PERIPHERAL FILE MAP
************************************************************/

/************************************************************
* SPECIAL FUNCTION REGISTER ADDRESSES + CONTROL BITS
************************************************************/

#define IE1_                (0x0000)  /* Interrupt Enable 1 */
sfrb    IE1               = IE1_;
#define WDTIE               (0x01)    /* Watchdog Interrupt Enable */
#define OFIE                (0x02)    /* Osc. Fault  Interrupt Enable */
#define NMIIE               (0x10)    /* NMI Interrupt Enable */
#define ACCVIE              (0x20)    /* Flash Access Violation Interrupt Enable */

#define IFG1_               (0x0002)  /* Interrupt Flag 1 */
sfrb    IFG1              = IFG1_;
#define WDTIFG              (0x01)    /* Watchdog Interrupt Flag */
#define OFIFG               (0x02)    /* Osc. Fault Interrupt Flag */
#define PORIFG              (0x04)    /* Power On Interrupt Flag */
#define RSTIFG              (0x08)    /* Reset Interrupt Flag */
#define NMIIFG              (0x10)    /* NMI Interrupt Flag */

#define IE2_                (0x0001)  /* Interrupt Enable 2 */
sfrb    IE2               = IE2_;
#define UCA0RXIE            (0x01)
#define UCA0TXIE            (0x02)
#define UCB0RXIE            (0x04)
#define UCB0TXIE            (0x08)

#define IFG2_               (0x0003)  /* Interrupt Flag 2 */
sfrb    IFG2              = IFG2_;
#define UCA0RXIFG           (0x01)
#define UCA0TXIFG           (0x02)
#define UCB0RXIFG           (0x04)
#define UCB0TXIFG           (0x08)

/************************************************************
* ADC10
************************************************************/
#define __MSP430_HAS_ADC10__            /* Definition to show that Module is available */

#define ADC10DTC0_          (0x0048)    /* ADC10 Data Transfer Control 0 */
sfrb    ADC10DTC0         = ADC10DTC0_;
#define ADC10DTC1_          (0x0049)    /* ADC10 Data Transfer Control 1 */
sfrb    ADC10DTC1         = ADC10DTC1_;
#define ADC10AE0_           (0x004A)    /* ADC10 Analog Enable 0 */
sfrb    ADC10AE0          = ADC10AE0_;
#define ADC10AE1_           (0x004B)    /* ADC10 Analog Enable 1 */
sfrb    ADC10AE1          = ADC10AE1_;

#define ADC10CTL0_          (0x01B0)    /* ADC10 Control 0 */
sfrw    ADC10CTL0         = ADC10CTL0_;
#define ADC10CTL1_          (0x01B2)    /* ADC10 Control 1 */
sfrw    ADC10CTL1         = ADC10CTL1_;
#define ADC10MEM_           (0x01B4)    /* ADC10 Memory */
sfrw    ADC10MEM          = ADC10MEM_;
#define ADC10SA_            (0x01BC)    /* ADC10 Data Transfer Start Address */
sfrw    ADC10SA           = ADC10SA_;

/* ADC10CTL0 */
#define ADC10SC             (0x001)     /* ADC10 Start Conversion */
#define ENC                 (0x002)     /* ADC10 Enable Conversion */
#define ADC10IFG            (0x004)     /* ADC10 Interrupt Flag */
#define ADC10IE             (0x008)     /* ADC10 Interrupt Enalbe */
#define ADC10ON             (0x010)     /* ADC10 On/Enable */
#define REFON               (0x020)     /* ADC10 Reference on */
#define REF2_5V             (0x040)     /* ADC10 Ref 0:1.5V / 1:2.5V */
#define MSC                 (0x080)     /* ADC10 Multiple SampleConversion */
#define REFBURST            (0x100)     /* ADC10 Reference Burst Mode */
#define REFOUT              (0x200)     /* ADC10 Enalbe output of Ref. */
#define ADC10SR             (0x400)     /* ADC10 Sampling Rate 0:200ksps / 1:50ksps */
#define ADC10SHT0           (0x800)     /* ADC10 Sample Hold Select 0 */
#define ADC10SHT1           (0x1000)    /* ADC10 Sample Hold Select 1 */
#define SREF0               (0x2000)    /* ADC10 Reference Select 0 */
#define SREF1               (0x4000)    /* ADC10 Reference Select 1 */
#define SREF3               (0x8000)    /* ADC10 Reference Select 2 */
#define ADC10SHT_0          (0*0x800)   /* 4 x ADC10CLKs */
#define ADC10SHT_1          (1*0x800)   /* 8 x ADC10CLKs */
#define ADC10SHT_2          (2*0x800)   /* 16 x ADC10CLKs */
#define ADC10SHT_3          (3*0x800)   /* 64 x ADC10CLKs */

#define SREF_0              (0*0x2000)  /* VR+ = AVCC and VR- = AVSS */
#define SREF_1              (1*0x2000)  /* VR+ = VREF+ and VR- = AVSS */
#define SREF_2              (2*0x2000)  /* VR+ = VEREF+ and VR- = AVSS */
#define SREF_3              (3*0x2000)  /* VR+ = VEREF+ and VR- = AVSS */
#define SREF_4              (4*0x2000)  /* VR+ = AVCC and VR- = VREF-/VEREF- */
#define SREF_5              (5*0x2000)  /* VR+ = VREF+ and VR- = VREF-/VEREF- */
#define SREF_6              (6*0x2000)  /* VR+ = VEREF+ and VR- = VREF-/VEREF- */
#define SREF_7              (7*0x2000)  /* VR+ = VEREF+ and VR- = VREF-/VEREF- */

/* ADC10CTL1 */
#define ADC10BUSY           (0x0001)    /* ADC10CTL1 */
#define CONSEQ0             (0x0002)    /* ADC10 Conversion Sequence Select 0 */
#define CONSEQ1             (0x0004)    /* ADC10 Conversion Sequence Select 1 */
#define ADC10SSEL0          (0x0008)    /* ADC10 Clock Source Select 0 */
#define ADC10SSEL1          (0x0010)    /* ADC10 Clock Source Select 1 */
#define ADC10DIV0           (0x0020)    /* ADC10 Clock Divider Select 0 */
#define ADC10DIV1           (0x0040)    /* ADC10 Clock Divider Select 1 */
#define ADC10DIV2           (0x0080)    /* ADC10 Clock Divider Select 2 */
#define ISSH                (0x0100)    /* ADC10 Invert Sample Hold Signal */
#define ADC10DF             (0x0200)    /* ADC10 Data Format 0:binary 1:2's complement */
#define SHS0                (0x0400)    /* ADC10 Sample/Hold Source 0 */
#define SHS1                (0x0800)    /* ADC10 Sample/Hold Source 1 */
#define INCH0               (0x1000)    /* ADC10 Input Channel Select 0 */
#define INCH1               (0x2000)    /* ADC10 Input Channel Select 1 */
#define INCH2               (0x4000)    /* ADC10 Input Channel Select 2 */
#define INCH3               (0x8000)    /* ADC10 Input Channel Select 3 */

#define CONSEQ_0            (0*2)       /* Single channel single conversion */
#define CONSEQ_1            (1*2)       /* Sequence of channels */
#define CONSEQ_2            (2*2)       /* Repeat single channel */
#define CONSEQ_3            (3*2)       /* Repeat sequence of channels */

#define ADC10SSEL_0         (0*8)       /* ADC10OSC */
#define ADC10SSEL_1         (1*8)       /* ACLK */
#define ADC10SSEL_2         (2*8)       /* MCLK */
#define ADC10SSEL_3         (3*8)       /* SMCLK */

#define ADC10DIV_0          (0*0x20)
#define ADC10DIV_1          (1*0x20)
#define ADC10DIV_2          (2*0x20)
#define ADC10DIV_3          (3*0x20)
#define ADC10DIV_4          (4*0x20)
#define ADC10DIV_5          (5*0x20)
#define ADC10DIV_6          (6*0x20)
#define ADC10DIV_7          (7*0x20)

#define SHS_0               (0*0x400)   /* ADC10SC */
#define SHS_1               (1*0x400)   /* TA3 OUT1 */
#define SHS_2               (2*0x400)   /* TA3 OUT0 */
#define SHS_3               (3*0x400)   /* TA3 OUT2 */

#define INCH_0              (0*0x1000)
#define INCH_1              (1*0x1000)
#define INCH_2              (2*0x1000)
#define INCH_3              (3*0x1000)
#define INCH_4              (4*0x1000)
#define INCH_5              (5*0x1000)
#define INCH_6              (6*0x1000)
#define INCH_7              (7*0x1000)
#define INCH_8              (8*0x1000)
#define INCH_9              (9*0x1000)
#define INCH_10             (10*0x1000)
#define INCH_11             (11*0x1000)
#define INCH_12             (12*0x1000) /* Selects Channel 11 */
#define INCH_13             (13*0x1000) /* Selects Channel 11 */
#define INCH_14             (14*0x1000) /* Selects Channel 11 */
#define INCH_15             (15*0x1000) /* Selects Channel 11 */

/* ADC10DTC0 */
#define ADC10FETCH          (0x001)
#define ADC10B1             (0x002)
#define ADC10CT             (0x004)
#define ADC10TB             (0x008)
#define ADC10DISABLE        (0x000)     /* ADC10DTC1 */

/************************************************************
* Basic Clock Module
************************************************************/
#define __MSP430_HAS_BC2__            /* Definition to show that Module is available */

#define DCOCTL_             (0x0056)  /* DCO Clock Frequency Control */
sfrb    DCOCTL            = DCOCTL_;
#define BCSCTL1_            (0x0057)  /* Basic Clock System Control 1 */
sfrb    BCSCTL1           = BCSCTL1_;
#define BCSCTL2_            (0x0058)  /* Basic Clock System Control 2 */
sfrb    BCSCTL2           = BCSCTL2_;
#define BCSCTL3_            (0x0053)  /* Basic Clock System Control 3 */
sfrb    BCSCTL3           = BCSCTL3_;

#define MOD0                (0x01)   /* Modulation Bit 0 */
#define MOD1                (0x02)   /* Modulation Bit 1 */
#define MOD2                (0x04)   /* Modulation Bit 2 */
#define MOD3                (0x08)   /* Modulation Bit 3 */
#define MOD4                (0x10)   /* Modulation Bit 4 */
#define DCO0                (0x20)   /* DCO Select Bit 0 */
#define DCO1                (0x40)   /* DCO Select Bit 1 */
#define DCO2                (0x80)   /* DCO Select Bit 2 */

#define RSEL0               (0x01)   /* Range Select Bit 0 */
#define RSEL1               (0x02)   /* Range Select Bit 1 */
#define RSEL2               (0x04)   /* Range Select Bit 2 */
#define RSEL3               (0x08)   /* Range Select Bit 3 */
#define DIVA0               (0x10)   /* ACLK Divider 0 */
#define DIVA1               (0x20)   /* ACLK Divider 1 */
#define XTS                 (0x40)   /* LFXTCLK 0:Low Freq. / 1: High Freq. */
#define XT2OFF              (0x80)   /* Enable XT2CLK */

#define DIVA_0              (0x00)   /* ACLK Divider 0: /1 */
#define DIVA_1              (0x10)   /* ACLK Divider 1: /2 */
#define DIVA_2              (0x20)   /* ACLK Divider 2: /4 */
#define DIVA_3              (0x30)   /* ACLK Divider 3: /8 */

#define DCOR                (0x01)   /* Enable External Resistor : 1 */
#define DIVS0               (0x02)   /* SMCLK Divider 0 */
#define DIVS1               (0x04)   /* SMCLK Divider 1 */
#define SELS                (0x08)   /* SMCLK Source Select 0:DCOCLK / 1:XT2CLK/LFXTCLK */
#define DIVM0               (0x10)   /* MCLK Divider 0 */
#define DIVM1               (0x20)   /* MCLK Divider 1 */
#define SELM0               (0x40)   /* MCLK Source Select 0 */
#define SELM1               (0x80)   /* MCLK Source Select 1 */

#define DIVS_0              (0x00)   /* SMCLK Divider 0: /1 */
#define DIVS_1              (0x02)   /* SMCLK Divider 1: /2 */
#define DIVS_2              (0x04)   /* SMCLK Divider 2: /4 */
#define DIVS_3              (0x06)   /* SMCLK Divider 3: /8 */

#define DIVM_0              (0x00)   /* MCLK Divider 0: /1 */
#define DIVM_1              (0x10)   /* MCLK Divider 1: /2 */
#define DIVM_2              (0x20)   /* MCLK Divider 2: /4 */
#define DIVM_3              (0x30)   /* MCLK Divider 3: /8 */

#define SELM_0              (0x00)   /* MCLK Source Select 0: DCOCLK */
#define SELM_1              (0x40)   /* MCLK Source Select 1: DCOCLK */
#define SELM_2              (0x80)   /* MCLK Source Select 2: XT2CLK/LFXTCLK */
#define SELM_3              (0xC0)   /* MCLK Source Select 3: LFXTCLK */
                                                                                   
#define LFXT1OF             (0x01)   /* Low/high Frequency Oscillator Fault Flag */
#define XT2OF               (0x02)   /* High frequency oscillator 2 fault flag */
#define XCAP0               (0x04)   /* XIN/XOUT Cap 0 */
#define XCAP1               (0x08)   /* XIN/XOUT Cap 1 */
#define LFXT1S0             (0x10)   /* Mode 0 for LFXT1 (XTS = 0) */
#define LFXT1S1             (0x20)   /* Mode 1 for LFXT1 (XTS = 0) */
#define XT2S0               (0x40)   /* Mode 0 for XT2 */
#define XT2S1               (0x80)   /* Mode 1 for XT2 */

#define XCAP_0              (0x00)   /* XIN/XOUT Cap : 0 pF */
#define XCAP_1              (0x04)   /* XIN/XOUT Cap : 6 pF */
#define XCAP_2              (0x08)   /* XIN/XOUT Cap : 10 pF */
#define XCAP_3              (0x0C)   /* XIN/XOUT Cap : 12.5 pF */

#define LFXT1S_0            (0x00)   /* Mode 0 for LFXT1 : Normal operation */
#define LFXT1S_1            (0x10)   /* Mode 1 for LFXT1 : Reserved */
#define LFXT1S_2            (0x20)   /* Mode 2 for LFXT1 : VLO */
#define LFXT1S_3            (0x30)   /* Mode 3 for LFXT1 : Digital input signal */

#define XT2S_0              (0x00)   /* Mode 0 for XT2 : 0.4 - 1 MHz */
#define XT2S_1              (0x40)   /* Mode 1 for XT2 : 1 - 4 MHz */
#define XT2S_2              (0x80)   /* Mode 2 for XT2 : 2 - 16 MHz */
#define XT2S_3              (0xC0)   /* Mode 3 for XT2 : Digital input signal */

/*************************************************************
* Flash Memory
*************************************************************/
#define __MSP430_HAS_FLASH2__         /* Definition to show that Module is available */

#define FCTL1_              (0x0128)  /* FLASH Control 1 */
sfrw    FCTL1             = FCTL1_;
#define FCTL2_              (0x012A)  /* FLASH Control 2 */
sfrw    FCTL2             = FCTL2_;
#define FCTL3_              (0x012C)  /* FLASH Control 3 */
sfrw    FCTL3             = FCTL3_;

#define FRKEY               (0x9600)  /* Flash key returned by read */
#define FWKEY               (0xA500)  /* Flash key for write */
#define FXKEY               (0x3300)  /* for use with XOR instruction */

#define ERASE               (0x0002)  /* Enable bit for Flash segment erase */
#define MERAS               (0x0004)  /* Enable bit for Flash mass erase */
#define EEI                 (0x0008)  /* Enable Erase Interrupts */
#define EEIEX               (0x0010)  /* Enable Emergency Interrupt Exit */
#define WRT                 (0x0040)  /* Enable bit for Flash write */
#define BLKWRT              (0x0080)  /* Enable bit for Flash segment write */
#define SEGWRT              (0x0080)  /* old definition */ /* Enable bit for Flash segment write */

#define FN0                 (0x0001)  /* Divide Flash clock by 1 to 64 using FN0 to FN5 according to: */
#define FN1                 (0x0002)  /*  32*FN5 + 16*FN4 + 8*FN3 + 4*FN2 + 2*FN1 + FN0 + 1 */
#ifndef FN2
#define FN2                 (0x0004)
#endif
#ifndef FN3
#define FN3                 (0x0008)
#endif
#ifndef FN4
#define FN4                 (0x0010)
#endif
#define FN5                 (0x0020)
#define FSSEL0              (0x0040)  /* Flash clock select 0 */        /* to distinguish from USART SSELx */
#define FSSEL1              (0x0080)  /* Flash clock select 1 */

#define FSSEL_0             (0x0000)  /* Flash clock select: 0 - ACLK */
#define FSSEL_1             (0x0040)  /* Flash clock select: 1 - MCLK */
#define FSSEL_2             (0x0080)  /* Flash clock select: 2 - SMCLK */
#define FSSEL_3             (0x00C0)  /* Flash clock select: 3 - SMCLK */

#define BUSY                (0x0001)  /* Flash busy: 1 */
#define KEYV                (0x0002)  /* Flash Key violation flag */
#define ACCVIFG             (0x0004)  /* Flash Access violation flag */
#define WAIT                (0x0008)  /* Wait flag for segment write */
#define LOCK                (0x0010)  /* Lock bit: 1 - Flash is locked (read only) */
#define EMEX                (0x0020)  /* Flash Emergency Exit */
#define LOCKA               (0x0040)  /* Segment A Lock bit: read = 1 - Segment is locked (read only) */
#define FAIL                (0x0080)  /* Last Program or Erase failed */

/************************************************************
* Operational Amplifier
************************************************************/
#define __MSP430_HAS_OA_2__           /* Definition to show that Module is available */

#define OA0CTL0_            (0x00C0)  /* OA0 Control register 0 */
sfrb    OA0CTL0           = OA0CTL0_;
#define OA0CTL1_            (0x00C1)  /* OA0 Control register 1 */
sfrb    OA0CTL1           = OA0CTL1_;
#define OA1CTL0_            (0x00C2)  /* OA1 Control register 0 */
sfrb    OA1CTL0           = OA1CTL0_;
#define OA1CTL1_            (0x00C3)  /* OA1 Control register 1 */
sfrb    OA1CTL1           = OA1CTL1_;

#define OAADC0              (0x01)    /* OAx output to ADC12 input channel select 0 */
#define OAADC1              (0x02)    /* OAx output to ADC12 input channel select 1 */
#define OAPM0               (0x04)    /* OAx Power mode select 0 */
#define OAPM1               (0x08)    /* OAx Power mode select 1 */
#define OAP0                (0x10)    /* OAx Non-inverting input select 0 */
#define OAP1                (0x20)    /* OAx Non-inverting input select 1 */
#define OAN0                (0x40)    /* OAx Inverting input select 0 */
#define OAN1                (0x80)    /* OAx Inverting input select 1 */

#define OAPM_0              (0x00)    /* OAx Power mode select: off */
#define OAPM_1              (0x04)    /* OAx Power mode select: slow */
#define OAPM_2              (0x08)    /* OAx Power mode select: meduim */
#define OAPM_3              (0x0C)    /* OAx Power mode select: fast */
#define OAP_0               (0x00)    /* OAx Non-inverting input select 00 */
#define OAP_1               (0x10)    /* OAx Non-inverting input select 01 */
#define OAP_2               (0x20)    /* OAx Non-inverting input select 10 */
#define OAP_3               (0x30)    /* OAx Non-inverting input select 11 */
#define OAN_0               (0x00)    /* OAx Inverting input select 00 */
#define OAN_1               (0x40)    /* OAx Inverting input select 01 */
#define OAN_2               (0x80)    /* OAx Inverting input select 10 */
#define OAN_3               (0xC0)    /* OAx Inverting input select 11 */

#define OARRIP              (0x01)    /* OAx Rail-to-Rail Input off */
#define OANEXT              (0x02)    /* OAx Inverting input external */
//#define OACAL               (0x02)    /* OAx Offset Calibration */
#define OAFC0               (0x04)    /* OAx Function control 0 */
#define OAFC1               (0x08)    /* OAx Function control 1 */
#define OAFC2               (0x10)    /* OAx Function control 2 */
#define OAFBR0              (0x20)    /* OAx Feedback resistor select 0 */
#define OAFBR1              (0x40)    /* OAx Feedback resistor select 1 */
#define OAFBR2              (0x80)    /* OAx Feedback resistor select 2 */

#define OAFC_0              (0x00)    /* OAx Function: Gen. Purpose */
#define OAFC_1              (0x04)    /* OAx Function: Unity gain buffer */
#define OAFC_2              (0x08)    /* OAx Function: Reserved */
#define OAFC_3              (0x0C)    /* OAx Function: Comparator */
#define OAFC_4              (0x10)    /* OAx Function: Non-inverting PGA */
#define OAFC_5              (0x14)    /* OAx Function: Cascaded non-inverting PGA */
#define OAFC_6              (0x18)    /* OAx Function: Inverting PGA */
#define OAFC_7              (0x1C)    /* OAx Function: Differential amplifier */
#define OAFBR_0             (0x00)    /* OAx Feedback resistor: Tap 0 */
#define OAFBR_1             (0x20)    /* OAx Feedback resistor: Tap 1 */
#define OAFBR_2             (0x40)    /* OAx Feedback resistor: Tap 2 */
#define OAFBR_3             (0x60)    /* OAx Feedback resistor: Tap 3 */
#define OAFBR_4             (0x80)    /* OAx Feedback resistor: Tap 4 */
#define OAFBR_5             (0xA0)    /* OAx Feedback resistor: Tap 5 */
#define OAFBR_6             (0xC0)    /* OAx Feedback resistor: Tap 6 */
#define OAFBR_7             (0xE0)    /* OAx Feedback resistor: Tap 7 */

/************************************************************
* DIGITAL I/O Port1/2 Pull up / Pull down Resistors
************************************************************/
#define __MSP430_HAS_PORT1_R__        /* Definition to show that Module is available */
#define __MSP430_HAS_PORT2_R__        /* Definition to show that Module is available */

#define P1IN_               (0x0020)  /* Port 1 Input */
const sfrb P1IN           = P1IN_;
#define P1OUT_              (0x0021)  /* Port 1 Output */
sfrb    P1OUT             = P1OUT_;
#define P1DIR_              (0x0022)  /* Port 1 Direction */
sfrb    P1DIR             = P1DIR_;
#define P1IFG_              (0x0023)  /* Port 1 Interrupt Flag */
sfrb    P1IFG             = P1IFG_;
#define P1IES_              (0x0024)  /* Port 1 Interrupt Edge Select */
sfrb    P1IES             = P1IES_;
#define P1IE_               (0x0025)  /* Port 1 Interrupt Enable */
sfrb    P1IE              = P1IE_;
#define P1SEL_              (0x0026)  /* Port 1 Selection */
sfrb    P1SEL             = P1SEL_;
#define P1REN_              (0x0027)  /* Port 1 Resistor Enable */
sfrb    P1REN             = P1REN_;

#define P2IN_               (0x0028)  /* Port 2 Input */
const sfrb P2IN           = P2IN_;
#define P2OUT_              (0x0029)  /* Port 2 Output */
sfrb    P2OUT             = P2OUT_;
#define P2DIR_              (0x002A)  /* Port 2 Direction */
sfrb    P2DIR             = P2DIR_;
#define P2IFG_              (0x002B)  /* Port 2 Interrupt Flag */
sfrb    P2IFG             = P2IFG_;
#define P2IES_              (0x002C)  /* Port 2 Interrupt Edge Select */
sfrb    P2IES             = P2IES_;
#define P2IE_               (0x002D)  /* Port 2 Interrupt Enable */
sfrb    P2IE              = P2IE_;
#define P2SEL_              (0x002E)  /* Port 2 Selection */
sfrb    P2SEL             = P2SEL_;
#define P2REN_              (0x002F)  /* Port 2 Resistor Enable */
sfrb    P2REN             = P2REN_;

/************************************************************
* DIGITAL I/O Port3/4 Pull up / Pull down Resistors
************************************************************/
#define __MSP430_HAS_PORT3_R__          /* Definition to show that Module is available */
#define __MSP430_HAS_PORT4_R__          /* Definition to show that Module is available */

#define P3IN_               (0x0018)  /* Port 3 Input */
const sfrb P3IN           = P3IN_;
#define P3OUT_              (0x0019)  /* Port 3 Output */
sfrb    P3OUT             = P3OUT_;
#define P3DIR_              (0x001A)  /* Port 3 Direction */
sfrb    P3DIR             = P3DIR_;
#define P3SEL_              (0x001B)  /* Port 3 Selection */
sfrb    P3SEL             = P3SEL_;
#define P3REN_              (0x0010)  /* Port 3 Resistor Enable */
sfrb    P3REN             = P3REN_;

#define P4IN_               (0x001C)  /* Port 4 Input */
const sfrb P4IN           = P4IN_;
#define P4OUT_              (0x001D)  /* Port 4 Output */
sfrb    P4OUT             = P4OUT_;
#define P4DIR_              (0x001E)  /* Port 4 Direction */
sfrb    P4DIR             = P4DIR_;
#define P4SEL_              (0x001F)  /* Port 4 Selection */
sfrb    P4SEL             = P4SEL_;
#define P4REN_              (0x0011)  /* Port 4 Resistor Enable */
sfrb    P4REN             = P4REN_;

/************************************************************
* Timer A3
************************************************************/
#define __MSP430_HAS_TA3__            /* Definition to show that Module is available */

#define TAIV_               (0x012E)  /* Timer A Interrupt Vector Word */
const sfrw TAIV           = TAIV_;
#define TACTL_              (0x0160)  /* Timer A Control */
sfrw    TACTL             = TACTL_;
#define TACCTL0_            (0x0162)  /* Timer A Capture/Compare Control 0 */
sfrw    TACCTL0           = TACCTL0_;
#define TACCTL1_            (0x0164)  /* Timer A Capture/Compare Control 1 */
sfrw    TACCTL1           = TACCTL1_;
#define TACCTL2_            (0x0166)  /* Timer A Capture/Compare Control 2 */
sfrw    TACCTL2           = TACCTL2_;
#define TAR_                (0x0170)  /* Timer A */
sfrw    TAR               = TAR_;
#define TACCR0_             (0x0172)  /* Timer A Capture/Compare 0 */
sfrw    TACCR0            = TACCR0_;
#define TACCR1_             (0x0174)  /* Timer A Capture/Compare 1 */
sfrw    TACCR1            = TACCR1_;
#define TACCR2_             (0x0176)  /* Timer A Capture/Compare 2 */
sfrw    TACCR2            = TACCR2_;

/* Alternate register names */
#define CCTL0               TACCTL0   /* Timer A Capture/Compare Control 0 */
#define CCTL1               TACCTL1   /* Timer A Capture/Compare Control 1 */
#define CCTL2               TACCTL2   /* Timer A Capture/Compare Control 2 */
#define CCR0                TACCR0    /* Timer A Capture/Compare 0 */
#define CCR1                TACCR1    /* Timer A Capture/Compare 1 */
#define CCR2                TACCR2    /* Timer A Capture/Compare 2 */
#define CCTL0_              TACCTL0_  /* Timer A Capture/Compare Control 0 */
#define CCTL1_              TACCTL1_  /* Timer A Capture/Compare Control 1 */
#define CCTL2_              TACCTL2_  /* Timer A Capture/Compare Control 2 */
#define CCR0_               TACCR0_   /* Timer A Capture/Compare 0 */
#define CCR1_               TACCR1_   /* Timer A Capture/Compare 1 */
#define CCR2_               TACCR2_   /* Timer A Capture/Compare 2 */

#define TASSEL1             (0x0200)  /* Timer A clock source select 0 */
#define TASSEL0             (0x0100)  /* Timer A clock source select 1 */
#define ID1                 (0x0080)  /* Timer A clock input divider 1 */
#define ID0                 (0x0040)  /* Timer A clock input divider 0 */
#define MC1                 (0x0020)  /* Timer A mode control 1 */
#define MC0                 (0x0010)  /* Timer A mode control 0 */
#define TACLR               (0x0004)  /* Timer A counter clear */
#define TAIE                (0x0002)  /* Timer A counter interrupt enable */
#define TAIFG               (0x0001)  /* Timer A counter interrupt flag */

#define MC_0                (0*0x10)  /* Timer A mode control: 0 - Stop */
#define MC_1                (1*0x10)  /* Timer A mode control: 1 - Up to CCR0 */
#define MC_2                (2*0x10)  /* Timer A mode control: 2 - Continous up */
#define MC_3                (3*0x10)  /* Timer A mode control: 3 - Up/Down */
#define ID_0                (0*0x40)  /* Timer A input divider: 0 - /1 */
#define ID_1                (1*0x40)  /* Timer A input divider: 1 - /2 */
#define ID_2                (2*0x40)  /* Timer A input divider: 2 - /4 */
#define ID_3                (3*0x40)  /* Timer A input divider: 3 - /8 */
#define TASSEL_0            (0*0x100) /* Timer A clock source select: 0 - TACLK */
#define TASSEL_1            (1*0x100) /* Timer A clock source select: 1 - ACLK  */
#define TASSEL_2            (2*0x100) /* Timer A clock source select: 2 - SMCLK */
#define TASSEL_3            (3*0x100) /* Timer A clock source select: 3 - INCLK */

#define CM1                 (0x8000)  /* Capture mode 1 */
#define CM0                 (0x4000)  /* Capture mode 0 */
#define CCIS1               (0x2000)  /* Capture input select 1 */
#define CCIS0               (0x1000)  /* Capture input select 0 */
#define SCS                 (0x0800)  /* Capture sychronize */
#define SCCI                (0x0400)  /* Latched capture signal (read) */
#define CAP                 (0x0100)  /* Capture mode: 1 /Compare mode : 0 */
#define OUTMOD2             (0x0080)  /* Output mode 2 */
#define OUTMOD1             (0x0040)  /* Output mode 1 */
#define OUTMOD0             (0x0020)  /* Output mode 0 */
#define CCIE                (0x0010)  /* Capture/compare interrupt enable */
#define CCI                 (0x0008)  /* Capture input signal (read) */
#define OUT                 (0x0004)  /* PWM Output signal if output mode 0 */
#define COV                 (0x0002)  /* Capture/compare overflow flag */
#define CCIFG               (0x0001)  /* Capture/compare interrupt flag */

#define OUTMOD_0            (0*0x20)  /* PWM output mode: 0 - output only */
#define OUTMOD_1            (1*0x20)  /* PWM output mode: 1 - set */
#define OUTMOD_2            (2*0x20)  /* PWM output mode: 2 - PWM toggle/reset */
#define OUTMOD_3            (3*0x20)  /* PWM output mode: 3 - PWM set/reset */
#define OUTMOD_4            (4*0x20)  /* PWM output mode: 4 - toggle */
#define OUTMOD_5            (5*0x20)  /* PWM output mode: 5 - Reset */
#define OUTMOD_6            (6*0x20)  /* PWM output mode: 6 - PWM toggle/set */
#define OUTMOD_7            (7*0x20)  /* PWM output mode: 7 - PWM reset/set */
#define CCIS_0              (0*0x1000) /* Capture input select: 0 - CCIxA */
#define CCIS_1              (1*0x1000) /* Capture input select: 1 - CCIxB */
#define CCIS_2              (2*0x1000) /* Capture input select: 2 - GND */
#define CCIS_3              (3*0x1000) /* Capture input select: 3 - Vcc */
#define CM_0                (0*0x4000) /* Capture mode: 0 - disabled */
#define CM_1                (1*0x4000) /* Capture mode: 1 - pos. edge */
#define CM_2                (2*0x4000) /* Capture mode: 1 - neg. edge */
#define CM_3                (3*0x4000) /* Capture mode: 1 - both edges */

/************************************************************
* Timer B3
************************************************************/
#define __MSP430_HAS_TB3__            /* Definition to show that Module is available */

#define TBIV_               (0x011E)  /* Timer B Interrupt Vector Word */
const sfrw TBIV           = TBIV_;
#define TBCTL_              (0x0180)  /* Timer B Control */
sfrw    TBCTL             = TBCTL_;
#define TBCCTL0_            (0x0182)  /* Timer B Capture/Compare Control 0 */
sfrw    TBCCTL0           = TBCCTL0_;
#define TBCCTL1_            (0x0184)  /* Timer B Capture/Compare Control 1 */
sfrw    TBCCTL1           = TBCCTL1_;
#define TBCCTL2_            (0x0186)  /* Timer B Capture/Compare Control 2 */
sfrw    TBCCTL2           = TBCCTL2_;
#define TBR_                (0x0190)  /* Timer B */
sfrw    TBR               = TBR_;
#define TBCCR0_             (0x0192)  /* Timer B Capture/Compare 0 */
sfrw    TBCCR0            = TBCCR0_;
#define TBCCR1_             (0x0194)  /* Timer B Capture/Compare 1 */
sfrw    TBCCR1            = TBCCR1_;
#define TBCCR2_             (0x0196)  /* Timer B Capture/Compare 2 */
sfrw    TBCCR2            = TBCCR2_;

#define TBCLGRP1            (0x4000)  /* Timer B Compare latch load group 1 */
#define TBCLGRP0            (0x2000)  /* Timer B Compare latch load group 0 */
#define CNTL1               (0x1000)  /* Counter lenght 1 */
#define CNTL0               (0x0800)  /* Counter lenght 0 */
#define TBSSEL1             (0x0200)  /* Clock source 1 */
#define TBSSEL0             (0x0100)  /* Clock source 0 */
#define TBCLR               (0x0004)  /* Timer B counter clear */
#define TBIE                (0x0002)  /* Timer B interrupt enable */
#define TBIFG               (0x0001)  /* Timer B interrupt flag */

#define SHR1                (0x4000)  /* Timer B Compare latch load group 1 */
#define SHR0                (0x2000)  /* Timer B Compare latch load group 0 */

#define TBSSEL_0            (0*0x0100)  /* Clock Source: TBCLK */
#define TBSSEL_1            (1*0x0100)  /* Clock Source: ACLK  */
#define TBSSEL_2            (2*0x0100)  /* Clock Source: SMCLK */
#define TBSSEL_3            (3*0x0100)  /* Clock Source: INCLK */
#define CNTL_0              (0*0x0800)  /* Counter lenght: 16 bit */
#define CNTL_1              (1*0x0800)  /* Counter lenght: 12 bit */
#define CNTL_2              (2*0x0800)  /* Counter lenght: 10 bit */
#define CNTL_3              (3*0x0800)  /* Counter lenght:  8 bit */
#define SHR_0               (0*0x2000)  /* Timer B Group: 0 - individually */
#define SHR_1               (1*0x2000)  /* Timer B Group: 1 - 3 groups (1-2, 3-4, 5-6) */
#define SHR_2               (2*0x2000)  /* Timer B Group: 2 - 2 groups (1-3, 4-6)*/
#define SHR_3               (3*0x2000)  /* Timer B Group: 3 - 1 group (all) */
#define TBCLGRP_0           (0*0x2000)  /* Timer B Group: 0 - individually */
#define TBCLGRP_1           (1*0x2000)  /* Timer B Group: 1 - 3 groups (1-2, 3-4, 5-6) */
#define TBCLGRP_2           (2*0x2000)  /* Timer B Group: 2 - 2 groups (1-3, 4-6)*/
#define TBCLGRP_3           (3*0x2000)  /* Timer B Group: 3 - 1 group (all) */

/* Additional Timer B Control Register bits are defined in Timer A */

#define CLLD1               (0x0400)  /* Compare latch load source 1 */
#define CLLD0               (0x0200)  /* Compare latch load source 0 */

#define SLSHR1              (0x0400)  /* Compare latch load source 1 */
#define SLSHR0              (0x0200)  /* Compare latch load source 0 */

#define SLSHR_0             (0*0x0200)  /* Compare latch load sourec : 0 - immediate */
#define SLSHR_1             (1*0x0200)  /* Compare latch load sourec : 1 - TBR counts to 0 */
#define SLSHR_2             (2*0x0200)  /* Compare latch load sourec : 2 - up/down */
#define SLSHR_3             (3*0x0200)  /* Compare latch load sourec : 3 - TBR counts to TBCTL0 */

#define CLLD_0              (0*0x0200)  /* Compare latch load sourec : 0 - immediate */
#define CLLD_1              (1*0x0200)  /* Compare latch load sourec : 1 - TBR counts to 0 */
#define CLLD_2              (2*0x0200)  /* Compare latch load sourec : 2 - up/down */
#define CLLD_3              (3*0x0200)  /* Compare latch load sourec : 3 - TBR counts to TBCTL0 */

/************************************************************
* USCI
************************************************************/
#define __MSP430_HAS_USCI__           /* Definition to show that Module is available */

#define UCA0CTL0_           (0x0060)  /* USCI A0 Control Register 0 */
sfrb    UCA0CTL0          = UCA0CTL0_;
#define UCA0CTL1_           (0x0061)  /* USCI A0 Control Register 1 */
sfrb    UCA0CTL1          = UCA0CTL1_;
#define UCA0BR0_            (0x0062)  /* USCI A0 Baud Rate 0 */
sfrb    UCA0BR0           = UCA0BR0_;
#define UCA0BR1_            (0x0063)  /* USCI A0 Baud Rate 1 */
sfrb    UCA0BR1           = UCA0BR1_;
#define UCA0MCTL_           (0x0064)  /* USCI A0 Modulation Control */
sfrb    UCA0MCTL          = UCA0MCTL_;
#define UCA0STAT_           (0x0065)  /* USCI A0 Status Register */
sfrb    UCA0STAT          = UCA0STAT_;
#define UCA0RXBUF_          (0x0066)  /* USCI A0 Receive Buffer */
const sfrb UCA0RXBUF      = UCA0RXBUF_;
#define UCA0TXBUF_          (0x0067)  /* USCI A0 Transmit Buffer */
sfrb    UCA0TXBUF         = UCA0TXBUF_;
#define UCA0ABCTL_          (0x005D)  /* USCI A0 LIN Control */
sfrb    UCA0ABCTL         = UCA0ABCTL_;
#define UCA0IRTCTL_         (0x005E)  /* USCI A0 IrDA Transmit Control */
sfrb    UCA0IRTCTL        = UCA0IRTCTL_;
#define UCA0IRRCTL_         (0x005F)  /* USCI A0 IrDA Receive Control */
sfrb    UCA0IRRCTL        = UCA0IRRCTL_;



#define UCB0CTL0_           (0x0068)  /* USCI B0 Control Register 0 */
sfrb    UCB0CTL0          = UCB0CTL0_;
#define UCB0CTL1_           (0x0069)  /* USCI B0 Control Register 1 */
sfrb    UCB0CTL1          = UCB0CTL1_;
#define UCB0BR0_            (0x006A)  /* USCI B0 Baud Rate 0 */
sfrb    UCB0BR0           = UCB0BR0_;
#define UCB0BR1_            (0x006B)  /* USCI B0 Baud Rate 1 */
sfrb    UCB0BR1           = UCB0BR1_;
#define UCB0I2CIE_          (0x006C)  /* USCI B0 I2C Interrupt Enable Register */
sfrb    UCB0I2CIE         = UCB0I2CIE_;
#define UCB0STAT_           (0x006D)  /* USCI B0 Status Register */
sfrb    UCB0STAT          = UCB0STAT_;
#define UCB0RXBUF_          (0x006E)  /* USCI B0 Receive Buffer */
const sfrb UCB0RXBUF      = UCB0RXBUF_;
#define UCB0TXBUF_          (0x006F)  /* USCI B0 Transmit Buffer */
sfrb    UCB0TXBUF         = UCB0TXBUF_;
#define UCB0I2COA_          (0x0118)  /* USCI B0 I2C Own Address */
sfrw    UCB0I2COA         = UCB0I2COA_;
#define UCB0I2CSA_          (0x011A)  /* USCI B0 I2C Slave Address */
sfrw    UCB0I2CSA         = UCB0I2CSA_;

// UART-Mode Bits
#define UCPEN               (0x80)    /* Async. Mode: Parity enable */
#define UCPAR               (0x40)    /* Async. Mode: Parity     0:odd / 1:even */
#define UCMSB               (0x20)    /* Async. Mode: MSB first  0:LSB / 1:MSB */
#define UC7BIT              (0x10)    /* Async. Mode: Data Bits  0:8-bits / 1:7-bits */
#define UCSPB               (0x08)    /* Async. Mode: Stop Bits  0:one / 1: two */
#define UCMODE1             (0x04)    /* Async. Mode: USCI Mode 1 */
#define UCMODE0             (0x02)    /* Async. Mode: USCI Mode 0 */
#define UCSYNC              (0x01)    /* Sync-Mode  0:UART-Mode / 1:SPI-Mode */

// SPI-Mode Bits
#define UCCKPH              (0x80)    /* Sync. Mode: Clock Phase */
#define UCCKPL              (0x40)    /* Sync. Mode: Clock Polarity */
#define UCMST               (0x08)    /* Sync. Mode: Master Select */

// I2C-Mode Bits
#define UCA10               (0x80)    /* 10-bit Address Mode */
#define UCSLA10             (0x40)    /* 10-bit Slave Address Mode */
#define UCMM                (0x20)    /* Multi-Master Environment */
//#define res               (0x10)    /* reserved */
#define UCMODE_0            (0x00)    /* Sync. Mode: USCI Mode: 0 */
#define UCMODE_1            (0x02)    /* Sync. Mode: USCI Mode: 1 */
#define UCMODE_2            (0x04)    /* Sync. Mode: USCI Mode: 2 */
#define UCMODE_3            (0x06)    /* Sync. Mode: USCI Mode: 3 */

// UART-Mode Bits
#define UCSSEL1             (0x80)    /* USCI 0 Clock Source Select 1 */
#define UCSSEL0             (0x40)    /* USCI 0 Clock Source Select 0 */
#define UCRXEIE             (0x20)    /* RX Error interrupt enable */
#define UCBRKIE             (0x10)    /* Break interrupt enable */                    
#define UCDORM              (0x08)    /* Dormant (Sleep) Mode */
#define UCTXADDR            (0x04)    /* Send next Data as Address */
#define UCTXBRK             (0x02)    /* Send next Data as Break */
#define UCSWRST             (0x01)    /* USCI Software Reset */

// SPI-Mode Bits
//#define res               (0x20)    /* reserved */
//#define res               (0x10)    /* reserved */
//#define res               (0x08)    /* reserved */
//#define res               (0x04)    /* reserved */
//#define res               (0x02)    /* reserved */

// I2C-Mode Bits
//#define res               (0x20)    /* reserved */
#define UCTR                (0x10)    /* Transmit/Receive Select/Flag */
#define UCTXNACK            (0x08)    /* Transmit NACK */
#define UCTXSTP             (0x04)    /* Transmit STOP */
#define UCTXSTT             (0x02)    /* Transmit START */
#define UCSSEL_0            (0x00)    /* USCI 0 Clock Source: 0 */
#define UCSSEL_1            (0x40)    /* USCI 0 Clock Source: 1 */
#define UCSSEL_2            (0x80)    /* USCI 0 Clock Source: 2 */
#define UCSSEL_3            (0xC0)    /* USCI 0 Clock Source: 3 */

#define UCBRF3              (0x80)    /* USCI First Stage Modulation Select 3 */
#define UCBRF2              (0x40)    /* USCI First Stage Modulation Select 2 */
#define UCBRF1              (0x20)    /* USCI First Stage Modulation Select 1 */
#define UCBRF0              (0x10)    /* USCI First Stage Modulation Select 0 */
#define UCBRS2              (0x08)    /* USCI Second Stage Modulation Select 2 */
#define UCBRS1              (0x04)    /* USCI Second Stage Modulation Select 1 */
#define UCBRS0              (0x02)    /* USCI Second Stage Modulation Select 0 */
#define UCOS16              (0x01)    /* USCI 16-times Oversampling enable */

#define UCBRF_0             (0x00)    /* USCI First Stage Modulation: 0 */
#define UCBRF_1             (0x10)    /* USCI First Stage Modulation: 1 */
#define UCBRF_2             (0x20)    /* USCI First Stage Modulation: 2 */
#define UCBRF_3             (0x30)    /* USCI First Stage Modulation: 3 */
#define UCBRF_4             (0x40)    /* USCI First Stage Modulation: 4 */
#define UCBRF_5             (0x50)    /* USCI First Stage Modulation: 5 */
#define UCBRF_6             (0x60)    /* USCI First Stage Modulation: 6 */
#define UCBRF_7             (0x70)    /* USCI First Stage Modulation: 7 */
#define UCBRF_8             (0x80)    /* USCI First Stage Modulation: 8 */
#define UCBRF_9             (0x90)    /* USCI First Stage Modulation: 9 */
#define UCBRF_10            (0xA0)    /* USCI First Stage Modulation: A */
#define UCBRF_11            (0xB0)    /* USCI First Stage Modulation: B */
#define UCBRF_12            (0xC0)    /* USCI First Stage Modulation: C */
#define UCBRF_13            (0xD0)    /* USCI First Stage Modulation: D */
#define UCBRF_14            (0xE0)    /* USCI First Stage Modulation: E */
#define UCBRF_15            (0xF0)    /* USCI First Stage Modulation: F */

#define UCBRS_0             (0x00)    /* USCI Second Stage Modulation: 0 */
#define UCBRS_1             (0x02)    /* USCI Second Stage Modulation: 1 */
#define UCBRS_2             (0x04)    /* USCI Second Stage Modulation: 2 */
#define UCBRS_3             (0x06)    /* USCI Second Stage Modulation: 3 */
#define UCBRS_4             (0x08)    /* USCI Second Stage Modulation: 4 */
#define UCBRS_5             (0x0A)    /* USCI Second Stage Modulation: 5 */
#define UCBRS_6             (0x0C)    /* USCI Second Stage Modulation: 6 */
#define UCBRS_7             (0x0E)    /* USCI Second Stage Modulation: 7 */

#define UCLISTEN            (0x80)    /* USCI Listen mode */
#define UCFE                (0x40)    /* USCI Frame Error Flag */
#define UCOE                (0x20)    /* USCI Overrun Error Flag */
#define UCPE                (0x10)    /* USCI Parity Error Flag */
#define UCBRK               (0x08)    /* USCI Break received */
#define UCRXERR             (0x04)    /* USCI RX Error Flag */
#define UCADDR              (0x02)    /* USCI Address received Flag */
#define UCBUSY              (0x01)    /* USCI Busy Flag */
#define UCIDLE              (0x02)    /* USCI Idle line detected Flag */

//#define res               (0x80)    /* reserved */
//#define res               (0x40)    /* reserved */
//#define res               (0x20)    /* reserved */
//#define res               (0x10)    /* reserved */
#define UCNACKIE            (0x08)    /* NACK Condition interrupt enable */
#define UCSTPIE             (0x04)    /* STOP Condition interrupt enable */
#define UCSTTIE             (0x02)    /* START Condition interrupt enable */
#define UCALIE              (0x01)    /* Arbitration Lost interrupt enable */

#define UCSCLLOW            (0x40)    /* SCL low */
#define UCGC                (0x20)    /* General Call address received Flag */
#define UCBBUSY             (0x10)    /* Bus Busy Flag */
#define UCNACKIFG           (0x08)    /* NAK Condition interrupt Flag */
#define UCSTPIFG            (0x04)    /* STOP Condition interrupt Flag */
#define UCSTTIFG            (0x02)    /* START Condition interrupt Flag */
#define UCALIFG             (0x01)    /* Arbitration Lost interrupt Flag */

#define UCIRTXPL5           (0x80)    /* IRDA Transmit Pulse Length 5 */
#define UCIRTXPL4           (0x40)    /* IRDA Transmit Pulse Length 4 */
#define UCIRTXPL3           (0x20)    /* IRDA Transmit Pulse Length 3 */
#define UCIRTXPL2           (0x10)    /* IRDA Transmit Pulse Length 2 */
#define UCIRTXPL1           (0x08)    /* IRDA Transmit Pulse Length 1 */
#define UCIRTXPL0           (0x04)    /* IRDA Transmit Pulse Length 0 */
#define UCIRTXCLK           (0x02)    /* IRDA Transmit Pulse Clock Select */
#define UCIREN              (0x01)    /* IRDA Encoder/Decoder enable */

#define UCIRRXFL5           (0x80)    /* IRDA Receive Filter Length 5 */
#define UCIRRXFL4           (0x40)    /* IRDA Receive Filter Length 4 */
#define UCIRRXFL3           (0x20)    /* IRDA Receive Filter Length 3 */
#define UCIRRXFL2           (0x10)    /* IRDA Receive Filter Length 2 */
#define UCIRRXFL1           (0x08)    /* IRDA Receive Filter Length 1 */
#define UCIRRXFL0           (0x04)    /* IRDA Receive Filter Length 0 */
#define UCIRRXPL            (0x02)    /* IRDA Receive Input Polarity */
#define UCIRRXFE            (0x01)    /* IRDA Receive Filter enable */

//#define res               (0x80)    /* reserved */
//#define res               (0x40)    /* reserved */
#define UCDELIM1            (0x20)    /* Break Sync Delimiter 1 */
#define UCDELIM0            (0x10)    /* Break Sync Delimiter 0 */
#define UCSTOE              (0x08)    /* Sync-Field Timeout error */
#define UCBTOE              (0x04)    /* Break Timeout error */
//#define res               (0x02)    /* reserved */
#define UCABDEN             (0x01)    /* Auto Baud Rate detect enable */

#define UCGCEN              (0x8000)  /* I2C General Call enable */
#define UCOA9               (0x0200)  /* I2C Own Address 9 */
#define UCOA8               (0x0100)  /* I2C Own Address 8 */
#define UCOA7               (0x0080)  /* I2C Own Address 7 */
#define UCOA6               (0x0040)  /* I2C Own Address 6 */
#define UCOA5               (0x0020)  /* I2C Own Address 5 */
#define UCOA4               (0x0010)  /* I2C Own Address 4 */
#define UCOA3               (0x0008)  /* I2C Own Address 3 */
#define UCOA2               (0x0004)  /* I2C Own Address 2 */
#define UCOA1               (0x0002)  /* I2C Own Address 1 */
#define UCOA0               (0x0001)  /* I2C Own Address 0 */

#define UCSA9               (0x0200)  /* I2C Slave Address 9 */
#define UCSA8               (0x0100)  /* I2C Slave Address 8 */
#define UCSA7               (0x0080)  /* I2C Slave Address 7 */
#define UCSA6               (0x0040)  /* I2C Slave Address 6 */
#define UCSA5               (0x0020)  /* I2C Slave Address 5 */
#define UCSA4               (0x0010)  /* I2C Slave Address 4 */
#define UCSA3               (0x0008)  /* I2C Slave Address 3 */
#define UCSA2               (0x0004)  /* I2C Slave Address 2 */
#define UCSA1               (0x0002)  /* I2C Slave Address 1 */
#define UCSA0               (0x0001)  /* I2C Slave Address 0 */

/************************************************************
* WATCHDOG TIMER
************************************************************/
#define __MSP430_HAS_WDT__            /* Definition to show that Module is available */

#define WDTCTL_             (0x0120)  /* Watchdog Timer Control */
sfrw    WDTCTL            = WDTCTL_;
/* The bit names have been prefixed with "WDT" */
#define WDTIS0              (0x0001)
#define WDTIS1              (0x0002)
#define WDTSSEL             (0x0004)
#define WDTCNTCL            (0x0008)
#define WDTTMSEL            (0x0010)
#define WDTNMI              (0x0020)
#define WDTNMIES            (0x0040)
#define WDTHOLD             (0x0080)

#define WDTPW               (0x5A00)

/* WDT-interval times [1ms] coded with Bits 0-2 */
/* WDT is clocked by fSMCLK (assumed 1MHz) */
#define WDT_MDLY_32         (WDTPW+WDTTMSEL+WDTCNTCL)                         /* 32ms interval (default) */
#define WDT_MDLY_8          (WDTPW+WDTTMSEL+WDTCNTCL+WDTIS0)                  /* 8ms     " */
#define WDT_MDLY_0_5        (WDTPW+WDTTMSEL+WDTCNTCL+WDTIS1)                  /* 0.5ms   " */
#define WDT_MDLY_0_064      (WDTPW+WDTTMSEL+WDTCNTCL+WDTIS1+WDTIS0)           /* 0.064ms " */
/* WDT is clocked by fACLK (assumed 32KHz) */
#define WDT_ADLY_1000       (WDTPW+WDTTMSEL+WDTCNTCL+WDTSSEL)                 /* 1000ms  " */
#define WDT_ADLY_250        (WDTPW+WDTTMSEL+WDTCNTCL+WDTSSEL+WDTIS0)          /* 250ms   " */
#define WDT_ADLY_16         (WDTPW+WDTTMSEL+WDTCNTCL+WDTSSEL+WDTIS1)          /* 16ms    " */
#define WDT_ADLY_1_9        (WDTPW+WDTTMSEL+WDTCNTCL+WDTSSEL+WDTIS1+WDTIS0)   /* 1.9ms   " */
/* Watchdog mode -> reset after expired time */
/* WDT is clocked by fSMCLK (assumed 1MHz) */
#define WDT_MRST_32         (WDTPW+WDTCNTCL)                                  /* 32ms interval (default) */
#define WDT_MRST_8          (WDTPW+WDTCNTCL+WDTIS0)                           /* 8ms     " */
#define WDT_MRST_0_5        (WDTPW+WDTCNTCL+WDTIS1)                           /* 0.5ms   " */
#define WDT_MRST_0_064      (WDTPW+WDTCNTCL+WDTIS1+WDTIS0)                    /* 0.064ms " */
/* WDT is clocked by fACLK (assumed 32KHz) */
#define WDT_ARST_1000       (WDTPW+WDTCNTCL+WDTSSEL)                          /* 1000ms  " */
#define WDT_ARST_250        (WDTPW+WDTCNTCL+WDTSSEL+WDTIS0)                   /* 250ms   " */
#define WDT_ARST_16         (WDTPW+WDTCNTCL+WDTSSEL+WDTIS1)                   /* 16ms    " */
#define WDT_ARST_1_9        (WDTPW+WDTCNTCL+WDTSSEL+WDTIS1+WDTIS0)            /* 1.9ms   " */

/* INTERRUPT CONTROL */
/* These two bits are defined in the Special Function Registers */
/* #define WDTIE               0x01 */
/* #define WDTIFG              0x01 */

/************************************************************
* Calibration Data in Info Mem
************************************************************/

#ifndef __DisableCalData

#define CALDCO_16MHZ_         (0x10F8)  /* DCOCTL  Calibration Data for 16MHz */
const sfrb CALDCO_16MHZ     = CALDCO_16MHZ_;
#define CALBC1_16MHZ_         (0x10F9)  /* BCSCTL1 Calibration Data for 16MHz */
const sfrb CALBC1_16MHZ     = CALBC1_16MHZ_;
#define CALDCO_12MHZ_         (0x10FA)  /* DCOCTL  Calibration Data for 12MHz */
const sfrb CALDCO_12MHZ     = CALDCO_12MHZ_;
#define CALBC1_12MHZ_         (0x10FB)  /* BCSCTL1 Calibration Data for 12MHz */
const sfrb CALBC1_12MHZ     = CALBC1_12MHZ_;
#define CALDCO_8MHZ_          (0x10FC)  /* DCOCTL  Calibration Data for 8MHz */
const sfrb CALDCO_8MHZ      = CALDCO_8MHZ_;
#define CALBC1_8MHZ_          (0x10FD)  /* BCSCTL1 Calibration Data for 8MHz */
const sfrb CALBC1_8MHZ      = CALBC1_8MHZ_;
#define CALDCO_1MHZ_          (0x10FE)  /* DCOCTL  Calibration Data for 1MHz */
const sfrb CALDCO_1MHZ      = CALDCO_1MHZ_;
#define CALBC1_1MHZ_          (0x10FF)  /* BCSCTL1 Calibration Data for 1MHz */
const sfrb CALBC1_1MHZ      = CALBC1_1MHZ_;

#endif /* #ifndef __DisableCalData */

/************************************************************
* Interrupt Vectors (offset from 0xFFE0)
************************************************************/

#define PORT1_VECTOR        (2 * 2)  /* 0xFFE4 Port 1 */
#define PORT2_VECTOR        (3 * 2)  /* 0xFFE6 Port 2 */
#define ADC10_VECTOR        (5 * 2)  /* 0xFFEA ADC10 */
#define USCIAB0TX_VECTOR    (6 * 2)  /* 0xFFEC USCI A0/B0 Transmit */
#define USCIAB0RX_VECTOR    (7 * 2)  /* 0xFFEE USCI A0/B0 Receive */
#define TIMERA1_VECTOR      (8 * 2)  /* 0xFFF0 Timer A CC1-2, TA */
#define TIMERA0_VECTOR      (9 * 2)  /* 0xFFF2 Timer A CC0 */
#define WDT_VECTOR          (10 * 2) /* 0xFFF4 Watchdog Timer */
#define TIMERB1_VECTOR      (12 * 2) /* 0xFFF8 Timer B CC1-2, TB */
#define TIMERB0_VECTOR      (13 * 2) /* 0xFFFA Timer B CC0 */
#define NMI_VECTOR          (14 * 2) /* 0xFFFC Non-maskable */
#define RESET_VECTOR        (15 * 2) /* 0xFFFE Reset [Highest Priority] */

/************************************************************
* End of Modules
************************************************************/

#endif /* #ifndef __msp430x22x4 */

