//******************************************************************************
//  MSP-FET430P410 Demo - Timer_A UART 9600 Echo, DCO SMCLK 
//
//  Description: Use timer_A CCR0 hardware output modes and SCCI data latch to
//  to implement UART function @ 9600 baud. Software does not directly read and
//  write to RX and TX pins, instead proper use of output modes and SCCI data
//  latch are demonstrated. Use of these hardware features eliminates ISR 
//  latency effects as hardware insures that output and input bit latching and 
//  timing are perfectly synchronized with timer_A regardless of other 
//  software activity. In the Mainloop the UART function readies the UART to 
//  receive one character and waits in LPM0 with all activity interrupt driven. 
//  After a character has been received, the UART receive function forces exit 
//  from LPM0 in the Mainloop which echo's back the received character.
//  ACLK = LFXT1 = 32768, MCLK = SMCLK = TACLK = DCO = 32xACLK = 1048576  
//  //*An external watch crystal is required on XIN XOUT for ACLK*//	  
//
//                MSP430F413
//            -----------------
//        /|\|              XIN|-        
//         | |                 | 32kHz     
//         --|RST          XOUT|-        
//           |                 |         
//           |    CCI0B/RX/P1.1|<--------  
//           |                 | 9600 8N1 
//           |    CCI0A/TX/P1.0|-------->


#define RXD   0x02                      // RXD on P1.1
#define TXD   0x01                      // TXD on P1.0

//  Conditions for 9600 Baud SW UART, SMCLK = 1048576

#define Bitime_5  0x47                  // ~ 0.5 bit length + small adjustment 
#define Bitime  0x6C                    // ~ 9620 baud
 
unsigned int RXTXData;
unsigned char BitCnt;

void TX_Byte (void);
void RX_Ready (void);

//  M.Buccini
//  Texas Instruments, Inc
//  March 2002
//******************************************************************************

#include <msp430x41x.h>

void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;             // Stop watchdog timer
  FLL_CTL0 |= XCAP14PF;                 // Configure load caps
  CCTL0 = OUT;                          // TXD Idle as Mark 
  TACTL = TASSEL1 + MC1;                // SMCLK, continous mode
  P1SEL = TXD + RXD;                    // PP1.0/1 TA0 for TXD/RXD function
  P1DIR = TXD;                          // TXD output on P1

// Mainloop
  for (;;)                              
  {
  RX_Ready();                           // UART ready to RX one Byte
  _BIS_SR(CPUOFF+GIE);                  // Enter LPM0 Until character RXed
  TX_Byte();                            // TX Back RXed Byte Received
  }
}


// Function Transmits Character from RXTXData Buffer
void TX_Byte (void)
{
  BitCnt = 0xA;                         // Load Bit counter, 8data + ST/SP
  CCR0 = TAR;                           // Current state of TA counter
  CCR0 += Bitime;                       // Some time till first bit
  RXTXData |= 0x100;                    // Add mark stop bit to RXTXData 
  RXTXData = RXTXData << 1;             // Add space start bit
  CCTL0 = OUTMOD0+CCIE;                 // TXD = mark = idle 
  while ( CCTL0 & CCIE );               // Wait for TX completion
}


// Function Readies UART to Receive Character into RXTXData Buffer
void RX_Ready (void)
{
  BitCnt = 0x8;                         // Load Bit counter
  CCTL0 = SCS+CCIS0+OUTMOD0+CM1+CAP+CCIE;   // Sync, Neg Edge, Capture          
}


// Timer A0 interrupt service routine
void Timer_A (void) __interrupt[TIMERA0_VECTOR] 
{
  CCR0 += Bitime;                       // Add Offset to CCR0

// RX
  if (CCTL0 & CCIS0)                    // RX on CCI0B?
  {
    if( CCTL0 & CAP )                   // Capture mode = start bit edge
    {
    CCTL0 &= ~ CAP;                     // Switch from capture to compare mode 
    CCR0 += Bitime_5;
    }
    else
    {
    RXTXData = RXTXData >> 1;           
      if (CCTL0 & SCCI)                 // Get bit waiting in receive latch
      RXTXData |= 0x80;                 
      BitCnt --;                        // All bits RXed?
      if ( BitCnt == 0)
//>>>>>>>>>> Decode of Received Byte Here <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
      {
      CCTL0 &= ~ CCIE;                  // All bits RXed, disable interrupt
      _BIC_SR_IRQ(CPUOFF);              // Clear LPM0 bits from 0(SR)
      }
//>>>>>>>>>> Decode of Received Byte Here <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
    }
  }
// TX
  else
  {
    if ( BitCnt == 0)
    CCTL0 &= ~ CCIE;                    // All bits TXed, disable interrupt
    else
    {
      CCTL0 |=  OUTMOD2;                  // TX Space
      if (RXTXData & 0x01)                 
      CCTL0 &= ~ OUTMOD2;                 // TX Mark
      RXTXData = RXTXData >> 1;
      BitCnt --;
    }
  }
}
