;* --------------------------------------------------------------------------------------
;*  Copyright (C) 2004 Dallas Semiconductor Corporation, All Rights Reserved.
;* 
;*  Permission is hereby granted, free of charge, to any person obtaining a
;*  copy of this software and associated documentation files (the "Software"),
;*  to deal in the Software without restriction, including without limitation
;*  the rights to use, copy, modify, merge, publish, distribute, sublicense,
;*  and/or sell copies of the Software, and to permit persons to whom the
;*  Software is furnished to do so, subject to the following conditions:
;* 
;*  The above copyright notice and this permission notice shall be included
;*  in all copies or substantial portions of the Software.
;* 
;*  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
;*  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
;*  MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
;*  IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES
;*  OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
;*  ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
;*  OTHER DEALINGS IN THE SOFTWARE.
;* 
;*  Except as contained in this notice, the name of Dallas Semiconductor
;*  shall not be used except as stated in the Dallas Semiconductor
;*  Branding Policy.
;*
;*  Module:        maxq2000ev_1407
;*  Description:   Utility functions for communicating with the MAX1407 that
;*                 is located on the MAXQ2000 EV Kit.
;*  Filename:      maxq2000ev_1407.asm
;*  Date:          November 2, 2004
;*  Version:       1.00
;*
;* --------------------------------------------------------------------------------------

#include "maxQ2000.inc"
#include "maxq2000_spi.inc"
#include "maxq2000ev_1407.inc"
.code
;*******************************************************************************
;* Function:  max1407_transmit
;*
;* Sends/Receives a character to the MAX1407.
;*
;* Input:     ACC - the character to send.
;*
;* Output:    ACC - the character received.
;*
;* Destroys:
;*
;*******************************************************************************
max1407_transmit::
    move    PO5.4, #0           ; Set /CS low.
    call    spi_transmit
    move    PO5.4, #1           ; Set /CS high.
    ret

;*******************************************************************************
;* Function:  max1407_init
;*
;* Sets the correct SPI modes for talking to the MAX1407, enables SPI, and
;* initializes the 1407.
;*
;* Input:     None.
;*
;* Output:    None.
;*
;* Destroys:  ACC, A[0], A[1], A[2], A[3], PSF
;*
;*******************************************************************************
max1407_init::

    move    A[0], #8000h		; System clock is 12288000 Hz
    move    A[1], #00BBh
    move    A[2], #0B20h		; Desired baud rate is 2100000 Hz
    move    A[3], #0020h
    call    spi_setbaudrate

    ; Set the appropriate modes for the 1407
    move    C, #SPI_IDLE_LOW
    call    spi_setclockpolarity
    move    C, #SPI_ACTIVE_EDGE
    call    spi_setclockphase
    move    C, #SPI_LENGTH_8
    call    spi_setcharacterlength
    move    C, #SPI_MASTER_MODE
    call    spi_setmode

    ; Enable SPI
    move    C, #1
    call    spi_enable

    ; Initialize the 1407
    move    ACC, #MAX1407REG_RUN
    sla
    or      #MAX1407_WRITE_MASK
    call    max1407_transmit

    ret

;*******************************************************************************
;* Function:  max1407_readregister
;*
;* Reads the contents of one of the MAX1407's registers.
;*
;* Input:     ACC  - Register to be read.
;*            A[1] - Size of the register to be read (SPI_LENGTH_8 or
;*                   SPI_LENGTH_16).
;*
;* Output:    ACC - The value read.
;*
;* Destroys:  PSF
;*
;*******************************************************************************
max1407_readregister::
    ; Send read register command to 1407.
    sla
    or      #MAX1407_READ_MASK
    move    C, #SPI_LENGTH_8
    call    spi_setcharacterlength
    call    max1407_transmit

    ; Send follow on byte to get the results.
    move    C, A[1].0
    call    spi_setcharacterlength
    move    ACC, #0
    call    max1407_transmit

    ; Restore the transfer length back to its defaults.
    move    C, #SPI_LENGTH_8
    call    spi_setcharacterlength

    ret

;*******************************************************************************
;* Function:  max1407_writeregister
;*
;* Writes a new value to one of the MAX1407's registers.
;*
;* Input:     ACC  - Register to be written.
;*            A[1] - Size of the register to be read (SPI_LENGTH_8 or
;*                   SPI_LENGTH_16).
;*            A[2] - New register contents.
;*
;* Output:    None.
;*
;* Destroys:  PSF
;*
;*******************************************************************************
max1407_writeregister::
    ; Send the write register command
    sla
    or      #MAX1407_WRITE_MASK
    move    C, #SPI_LENGTH_8
    call    spi_setcharacterlength
    call    max1407_transmit
    
    ; Send a follow on byte that contains the new data.
    move    C, A[1].0
    call    spi_setcharacterlength
    move    ACC, A[2]
    call    max1407_transmit
    
    ; Restore the transfer length back to its default.
    move    C, #SPI_LENGTH_8
    call    spi_setcharacterlength
    ret

end
