;* --------------------------------------------------------------------------------------
;*  Copyright (C) 2004 Dallas Semiconductor Corporation, All Rights Reserved.
;* 
;*  Permission is hereby granted, free of charge, to any person obtaining a
;*  copy of this software and associated documentation files (the "Software"),
;*  to deal in the Software without restriction, including without limitation
;*  the rights to use, copy, modify, merge, publish, distribute, sublicense,
;*  and/or sell copies of the Software, and to permit persons to whom the
;*  Software is furnished to do so, subject to the following conditions:
;* 
;*  The above copyright notice and this permission notice shall be included
;*  in all copies or substantial portions of the Software.
;* 
;*  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
;*  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
;*  MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
;*  IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES
;*  OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
;*  ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
;*  OTHER DEALINGS IN THE SOFTWARE.
;* 
;*  Except as contained in this notice, the name of Dallas Semiconductor
;*  shall not be used except as stated in the Dallas Semiconductor
;*  Branding Policy.
;*
;*  Module:        rtc
;*  Description:   Convenience Real-Time-Clock routines 
;*  Filename:      maxq2000_rtc.asm
;*  Date:          October 13, 2004
;*  Version:       1.00
;*
;*  Contains some convenience functions for communicating with the real-time clock of the
;*  MAXQ2000.  Note that the following functions do not depend on the Evaluation Kit, and
;*  may be used in any MAXQ2000 system.
;*
;* --------------------------------------------------------------------------------------
#include "maxQ2000.inc"
  .code
;********************************************************************
;*
;* Function:    init_rtc
;*
;* Description: Initialize the real time clock.  Turn on the 32 kHz 
;*              crystal and start the clock.
;*
;* Input:       None
;*
;* Output:      None
;*
;* Destroys:    PSF
;*
;********************************************************************
init_rtc::
    move  RCNT, #0                     ; start the 32 kHz
    call  wait_for_rtc                 ; wait for clock to be not-busy
    move  RCNT, #8000h                 ; write-enable the RTC enable bit
    move  RCNT, #8001h                 ; start the real-time-clock
    move  RCNT, #0001h                 ; turn of write access to RTC enable
    ret

;********************************************************************
;*
;* Function:    wait_for_rtc
;*
;* Description: Wait for the RTC to not report busy.
;*
;* Input:       None
;*
;* Output:      None
;*
;* Destroys:    PSF
;*
;********************************************************************
wait_for_rtc::
    move  c, RCNT.3                    ; grab the busy bit
    jump  c, wait_for_rtc              ; wait until it goes low
    ret

;********************************************************************
;*
;* Function:    rtc_setsecondsalarm
;*
;* Description: Sets the time-of-day alarm for the real-time-clock.
;*
;* Input:       A[2]:A[1] - value to write to the time-of-day alarm
;*
;* Output:      None
;*
;* Destroys:    PSF
;*
;********************************************************************
rtc_setsecondsalarm::
    call  wait_for_rtc
    move  RCNT.1, #0                   ; disable time of day alarm
    call  wait_for_rtc
    move  RASL, A[1]
    move  RASH, A[2]
    call  wait_for_rtc
    move  RCNT.1, #1                   ; enable the time of day alarm
    ret

;********************************************************************
;*
;* Function:    rtc_getseconds
;*
;* Description: Get the time-of-day value for the real-time-clock.
;*
;* Input:       None
;*
;* Output:      A[2]:A[1] - current value in the 'seconds' registers
;*
;* Destroys:    PSF
;*
;********************************************************************
rtc_getseconds::
    move  A[2], RTSH
    move  A[1], RTSL
    ret

;********************************************************************
;*
;* Function:    rtc_setseconds
;*
;* Description: Sets the time-of-day value for the real-time-clock.
;*              Disables the time-of-day alarm.
;*
;* Input:       A[2]:A[1] - value to write to the 'seconds' registers
;*
;* Output:      None
;*
;* Destroys:    PSF
;*
;********************************************************************
rtc_setseconds::
    call  wait_for_rtc
    move  RCNT, #8001h                 ; set the write enable bit
    call  wait_for_rtc
    move  RCNT, #8000h                 ; disable the real time clock
    call  wait_for_rtc
    move  RTSH, A[2]
    call  wait_for_rtc
    move  RTSL, A[1]
    call  wait_for_rtc
    move  RCNT, #8001h                 ; enable the real time clock
    move  RCNT, #0001h                 ; clear the write-enable bit
    ret

;********************************************************************
;*
;* Function:    rtc_translate
;*
;* Description: Convenience function to translate from a number of 
;*              seconds into days, hours, minutes, and seconds.
;*
;* Input:       A[1]:A[0] - input number of seconds
;*
;* Output:      A[11] = number of days
;*              A[12] = number of hours
;*              A[13] = number of minutes
;*              A[14] = number of seconds
;*
;* Destroys:    A[10:0], PSF, LC[0], AP/APC=0
;*
;********************************************************************
rtc_translate:
    move  A[3], #(86400 >> 16)         ; high number of seconds in a day
    move  A[2], #(86400 & 0FFFFh)      ; low number of seconds in a day
    call  Div32                        ; A[3:2] = seconds left today, A[1:0] = number of days
    move  A[11], A[0]                  ; store off number of days
    move  A[1], A[3]                   ; load up number of seconds left today
    move  A[0], A[2]                   ;   "    "    "    "    "    "    "
    move  A[3], #0                     ; high number of hours in a day
    move  A[2], #24                    ; low number of hours in a day
    call  Div32                        ; A[3:2] = seconds left this hour, A[1:0] = number of hours
    move  A[12], A[0]                  ; store off number of hours
    move  A[1], A[3]                   ; load number of seconds left this hour
    move  A[0], A[2]                   ;   "    "    "    "    "    "    "
    move  A[3], #0                     ; high number of minutes in an hour
    move  A[2], #60                    ; low number of minutes in an hour
    call  Div32                        ; A[3:2] = seconds left this minute, A[1:0] = number of minutes
    move  A[13], A[0]                  ; store off number of minutes
    move  A[14], A[2]                  ; store off number of seconds
    ret

end





