/*****************************************************************************
 * ARM Loader - Main Loop                                                    *
 *                                                                           *
 * Copyright (c) 2003 Rowley Associates Limited.                             *
 *                                                                           *
 * This file may be distributed under the terms of the License Agreement     *
 * provided with this software.                                              *
 *                                                                           *
 * THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE   *
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. *
 *****************************************************************************/

#include <libarm.h>
#include "__armcps.h"
#include "libmem.h"
#include "libmem_loader.h"

#ifdef __BIG_ENDIAN
static void
swapword(unsigned char *w)
{             
  unsigned char tmp;
  tmp = w[0]; 
  w[0] = w[3];
  w[3] = tmp; 
  tmp = w[1]; 
  w[1] = w[2];
  w[2] = tmp;
}       
#endif

void 
libmem_dcc_loader_start(libmem_dcc_loader_set_param_fn_t set_param_fn)
{
  for (;;)
    {
      unsigned int ctrl, length;
      ctrl = libarm_dcc_read(); 
      length = ctrl >> __CPS_CTRL_LENGTH_OFFSET;
      switch(ctrl & __CPS_CTRL_TAG_MASK)
        {
          case __CPS_TAG_POKE:
            {
              unsigned char *address = (unsigned char *)libarm_dcc_read();
              libmem_driver_handle_t *h = libmem_get_driver(address);
              int res = LIBMEM_STATUS_SUCCESS;
              if (h)
                {
                  while (length)
                    {
                      unsigned long data[16];
                      unsigned int frag = length > sizeof(data) ? sizeof(data) : length;
                      unsigned int frag_words = (frag + 3) / 4;
                      unsigned int i;
                      for (i = 0; i < frag_words; ++i)
                        {
                          data[i] = libarm_dcc_read();
#ifdef __BIG_ENDIAN
                          swapword((unsigned char *)data + i);                          
#endif
                        }
                      if (res == LIBMEM_STATUS_SUCCESS)
                        {
                          res = h->driver_functions->write(h, address, (unsigned char *)data, frag);
                          address += frag;
                        }
                      length -= frag;
                    }
                }
              else
                res = LIBMEM_STATUS_NO_DRIVER;
              libarm_dcc_write(res);          
              break;
            }
          case __CPS_TAG_PEEK:
            {
              unsigned char *address = (unsigned char *)libarm_dcc_read();
              libmem_flush();
              while (length)
                { 
                  int i;
                  unsigned int data;
                  for (i = 4; i; --i)
                    {
                      data >>= 8;
                      if (length)
                        {
                          data |= *address++ << 24;
                          --length;
                        }                      
                    }
                  libarm_dcc_write(data);
                }
              libarm_dcc_write(LIBMEM_STATUS_SUCCESS);
              break;
            }
          case __CPS_TAG_MEMSET:
            libarm_dcc_write(libmem_fill((unsigned char *)libarm_dcc_read(), (unsigned char)libarm_dcc_read(), length));
            break;
          case __CPS_TAG_ERASE:
            {
              unsigned char *address = (unsigned char *)libarm_dcc_read();
              unsigned char *erase_address = address;
              size_t erase_length = length;
              int res = libmem_erase(address, length, &erase_address, &erase_length);
#ifndef NO_AUTO_UNLOCK
              if (res == LIBMEM_STATUS_LOCKED)
                {
                  libmem_unlock(erase_address, erase_length);
                  res = libmem_erase(erase_address, erase_length, 0, 0);
                }
#endif
              libarm_dcc_write(2); // NOTE: A response value > 1 indicates extended erase bytes result information */
              libarm_dcc_write(res);
              libarm_dcc_write((unsigned int)erase_address);
              libarm_dcc_write(erase_length);
              break;
            }
          case __CPS_TAG_ERASE_ALL:
            {
              int res = libmem_erase_all();
#ifndef NO_AUTO_UNLOCK
              if (res == LIBMEM_STATUS_LOCKED)
                {
                  libmem_unlock_all();
                  res = libmem_erase_all();
                }
#endif
            libarm_dcc_write(res);
              break;
            }
          case __CPS_TAG_SET_PARAM:
            {
              int res = 0;
              unsigned int parameter = libarm_dcc_read();
              unsigned int value = libarm_dcc_read();
              if (set_param_fn)
                res = set_param_fn(parameter, value);
              libarm_dcc_write(res);
            }
            break;
          case __CPS_TAG_CRC32:
            libmem_flush();
            libarm_dcc_write(~libmem_crc32((const unsigned char *)libarm_dcc_read(), length, 0xFFFFFFFF));
            break;
          case __CPS_TAG_TERMINATE:
            libmem_flush();
            libarm_dcc_write(1);
            return;
        }
    }
}

