// Copyright (c) 2006 Rowley Associates Limited.
//
// This file may be distributed under the terms of the License Agreement
// provided with this software.
//
// THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#ifndef libmem_h
#define libmem_h

#ifdef __cplusplus
extern "C" {
#endif

#include <stddef.h>
#include <stdint.h>

/*! \file libmem.h
 *
 *  \brief
 *    libmem - The RAL memory driver library.
 *
 *  \section INTRODUCTION Introduction
 *    The aim of libmem is to provide a common programming interface for a wide
 *    range of different memory types.
 *
 *    libmem consists of a mechanism for installing drivers for the different
 *    memories and a set of common memory access and control functions that
 *    locate the driver for a particular memory range and call the appropriate
 *    memory driver functions for the operation. 
 *
 *    The libmem library also includes a set of memory drivers for common memory
 *    devices.
 *
 *  \section USING_LIBMEM Using libmem
 *
 *    The best way to demonstrate using libmem is to see it in use.
 *    The following example demonstrates copying a block of data into FLASH
 *    using a libmem common flash interface (CFI) driver.
 *
 *  \code
 *
 *  int libmem_example1(void)
 *  { 
 *    const int flash1_max_geometry_regions = 4;
 *    libmem_driver_handle_t  flash1_handle;
 *    libmem_geometry_t flash1_geometry[flash1_max_geometry_regions];
 *    libmem_flash_info_t flash1_info;
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    uint8_t *write_dest = flash1_start + 16; 
 *    const uint8_t write_data[8] = { 1, 2, 3, 4, 5, 6, 7, 8 };
 *    int res;
 *
 *    // Register the FLASH libmem driver
 *    res = libmem_register_cfi_driver(&flash1_handle, flash1_start, flash1_geometry, flash1_max_geometry_regions, &flash1_info);
 *    if (res != LIBMEM_STATUS_SUCCESS)
 *      return 0;
 *
 *    // Unlock the destination memory area.
 *    res = libmem_unlock(write_dest, sizeof(write_data));
 *    if (res != LIBMEM_STATUS_SUCCESS)
 *      return 0;
 *
 *    // Erase the destination memory area.
 *    res = libmem_erase(write_dest, sizeof(write_data), 0, 0);
 *    if (res != LIBMEM_STATUS_SUCCESS)
 *      return 0;
 *
 *    // Copy write_data to the destination memory area.
 *    res = libmem_write(write_dest, write_data, sizeof(write_data));
 *    if (res != LIBMEM_STATUS_SUCCESS)
 *      return 0;
 *
 *    // Complete any outstanding transactions and put FLASH memory back into read mode.
 *    res = libmem_flush();
 *    if (res != LIBMEM_STATUS_SUCCESS)
 *      return 0;
 *
 *    return 1;
 *  }
 *
 *  \endcode
 *
 *  The following section describes each of the libmem calls in the preceeding
 *  example in detail.
 *
 *  Before any memory operations can be carried out the libmem drivers that
 *  you are going to use must be registered. The following code registers
 *  a libmem CFI driver for a FLASH device located at the memory location
 *  pointed to by \a flash1_start.
 *
 *  \code
 *    // Register the FLASH libmem driver
 *    res = libmem_register_cfi_driver(&flash1_handle, flash1_start, flash1_geometry, flash1_max_geometry_regions, &flash1_info);
 *    if (res != LIBMEM_STATUS_SUCCESS)
 *      return 0;
 *  \endcode
 *
 *  This call attempts to detect the type of FLASH and register the correct
 *  libmem CFI driver based on the CFI information read out from the FLASH
 *  device. Note that as well as detecting the type of FLASH you have, use 
 *  of this function will link in all libmem driver so in your own code
 *  you may wish to save memory registering a specific CFI driver and 
 *  supplying the FLASH size and geometry information yourself.
 *
 *  For each driver you register you must allocate \a libmem_driver_handle_t
 *  structure to act as a handle for the driver. Will the full version of
 *  of libmem you can register as many drivers as you wish, if you are using
 *  the "light" version of libmem you can only register one driver.
 * 
 *  Once you have registered your drivers you can use the general libmem
 *  memory functions to access and control your memory. The starting address
 *  passed to these functions is used to decide which driver to use for the
 *  memory operation, operations cannot span multiple drivers.
 *  
 *  The next operation the example code carries out it to unlock the FLASH 
 *  in preperation for the erase and write operations.
 *  Unlocking is not necessary on all memory devices and this operation
 *  is not implemented in all libmem drivers.
 *
 *  \code
 *    // Unlock the destination memory area.
 *    res = libmem_unlock(write_dest, sizeof(write_data));
 *    if (res != LIBMEM_STATUS_SUCCESS)
 *      return 0;
 *  \endcode
 *
 *  Once the memory has been unlocked the FLASH memory is erased. Once again
 *  unlocking is not necessary on all memory devices and this operation may
 *  not be implemented in all libmem drivers.
 *
 *  \code
 *    // Erase the destination memory area.
 *    res = libmem_erase(write_dest, sizeof(write_data), 0, 0);
 *    if (res != LIBMEM_STATUS_SUCCESS)
 *      return 0;
 *  \endcode
 *
 *  Parameters three and four of \a libmem_erase are not used in this example,
 *  however they provide a mechanism to allow the caller to determine how
 *  much memory was actually erased by the erase operation as it may well
 *  be more than requested.
 *
 *  Once the FLASH memory has been erased the FLASH can be programmed using
 *  the \a libmem_write function.
 *
 *  \code
 *    // Copy write_data to the destination memory area.
 *    res = libmem_write(write_dest, write_data, sizeof(write_data));
 *    if (res != LIBMEM_STATUS_SUCCESS)
 *      return 0;
 *  \endcode
 *
 *  The final step is to call \a libmem_flush. Once again flushing is not
 *  necessary on all memory devices, but some libmem drivers do not necessarily
 *  carry out operations immediately or they may leave the memory in an unreadable
 *  state for performance reasons and calling \a libmem_flush is required
 *  to flush outstanding operations and return the device to read mode.
 *
 *  \code
 *    // Complete any outstanding transactions and put FLASH memory back into read mode.
 *    res = libmem_flush();
 *    if (res != LIBMEM_STATUS_SUCCESS)
 *      return 0;
 *  \endcode
 *
 *  \section WRITING_LIBMEM Writing libmem drivers
 *
 *    libmem includes a set of memory drivers for common memory devices which
 *    means in most cases you probably won't need to write a libmem driver. If
 *    however you wish to use libmem to drive other unsupported memory devices
 *    you will need to write your own libmem driver.
 * 
 *
 */

/*! \def LIBMEM_STATUS_SUCCESS
 *
 *  Status result returned from libmem functions indicating success.
 */
#define LIBMEM_STATUS_SUCCESS                    (1)


/*! \def LIBMEM_STATUS_ERROR
 *
 *  Status result returned from libmem functions indicating a non-specific
 *  error.
 */
#define LIBMEM_STATUS_ERROR                      (0)


/*! \def LIBMEM_STATUS_TIMEOUT
 *
 *  Status result returned from libmem functions indicating that the operation
 *  has timed out.
 */
#define LIBMEM_STATUS_TIMEOUT                    (-1)


/*! \def LIBMEM_STATUS_LOCKED
 *
 *  Status result returned from libmem functions indicating that the operation
 *  could not be completed because the memory is locked.
 */
#define LIBMEM_STATUS_LOCKED                     (-2)


/*! \def LIBMEM_STATUS_NOT_IMPLEMENTED
 *
 *  Status result returned from libmem functions indicating that the operation
 *  being carried out has not been implemented in the libmem driver.
 */
#define LIBMEM_STATUS_NOT_IMPLEMENTED            (-3)


/*! \def LIBMEM_STATUS_GEOMETRY_REGION_OVERFLOW
 *
 *  Status result returned from libmem functions indicating that not enough room
 *  to store all the geometry region information.
 */
#define LIBMEM_STATUS_GEOMETRY_REGION_OVERFLOW   (-4)


/*! \def LIBMEM_STATUS_NO_DRIVER
 *
 *  Status result returned from libmem functions indicating that no driver has
 *  been installed for the region of memory being used.
 */
#define LIBMEM_STATUS_NO_DRIVER                  (-5)


/*! \def LIBMEM_STATUS_CFI_ERROR
 *
 *  Status result returned from libmem functions indicating that an error has
 *  been detected reading out the CFI information.
 */
#define LIBMEM_STATUS_CFI_ERROR                  (-6)


/*! \def LIBMEM_STATUS_INVALID_RANGE
 *
 *  Status result returned from libmem functions indicating that an invalid
 *  address range has been passed to the function.
 */
#define LIBMEM_STATUS_INVALID_RANGE              (-7)


/*! \def LIBMEM_STATUS_INVALID_PARAMETER
 *
 *  Status result returned from libmem functions indicating that an invalid
 *  parameter has been passed to the function.
 */
#define LIBMEM_STATUS_INVALID_PARAMETER          (-8)


/* 
 * CFI command set numbers.
 */

#define LIBMEM_CFI_CMDSET_NONE                   (0x0000)
#define LIBMEM_CFI_CMDSET_INTEL_EXTENDED         (0x0001)
#define LIBMEM_CFI_CMDSET_AMD_STANDARD           (0x0002)
#define LIBMEM_CFI_CMDSET_INTEL_STANDARD         (0x0003)
#define LIBMEM_CFI_CMDSET_AMD_EXTENDED           (0x0004)
#define LIBMEM_CFI_CMDSET_WINBOND_STANDARD       (0x0006)
#define LIBMEM_CFI_CMDSET_MITSUBISHI_STANDARD    (0x0100)
#define LIBMEM_CFI_CMDSET_MITSUBISHI_EXTENDED    (0x0101)
#define LIBMEM_CFI_CMDSET_SST_PAGE_WRITE         (0x0102)
#define LIBMEM_CFI_CMDSET_INTEL_PERFORMANCE_CODE (0x0200)
#define LIBMEM_CFI_CMDSET_INTEL_DATA             (0x0210)
#define LIBMEM_CFI_CMDSET_RESERVED               (0xFFFF)


/*! \def LIBMEM_ADDRESS_IN_RANGE(address, startAddress, endAddress)
 *
 *  \brief
 *    Macro to determine whether an address is within an address range.
 *
 *  \param address
 *    The address to check.
 *
 *  \param startAddress
 *    The start address of the address range.
 *
 *  \param endAddress
 *    The end address of the address range.
 *
 *  \return
 *    Non-zero if address is within address range.
 */
#define LIBMEM_ADDRESS_IN_RANGE(address, startAddress, endAddress) ((address >= startAddress) && (address <= endAddress))


/*! \def LIBMEM_RANGE_WITHIN_RANGE(r1StartAddress, r1EndAddress, r2StartAddress, r2EndAddress)
 *
 *  \brief
 *    Macro to determine whether an address range 1 is within address range 2.
 *
 *  \param r1StartAddress
 *    The start address of address range 1.
 *
 *  \param r1EndAddress
 *    The end address of address range 1.
 *
 *  \param r2StartAddress
 *    The start address of address range 2.
 *
 *  \param r2EndAddress
 *    The end address of address range 2.
 *
 *  \return
 *    Non-zero if address range 1 is within address range 2.
 */
#define LIBMEM_RANGE_WITHIN_RANGE(r1StartAddress, r1EndAddress, r2StartAddress, r2EndAddress) (LIBMEM_ADDRESS_IN_RANGE(r1StartAddress, r2StartAddress, r2EndAddress) && LIBMEM_ADDRESS_IN_RANGE(r1EndAddress, r2StartAddress, r2EndAddress))


/*! \def LIBMEM_RANGE_OVERLAPS_RANGE(r1StartAddress, r1EndAddress, r2StartAddress, r2EndAddress)
 *
 *  \brief
 *    Macro to determine whether an address range 1 overlaps address range 2.
 *
 *  \param r1StartAddress
 *    The start address of address range 1.
 *
 *  \param r1EndAddress
 *    The end address of address range 1.
 *
 *  \param r2StartAddress
 *    The start address of address range 2.
 *
 *  \param r2EndAddress
 *    The end address of address range 2.
 *
 *  \return
 *    Non-zero if address range 1 overlaps address range 2.
 */
#define LIBMEM_RANGE_OVERLAPS_RANGE(r1StartAddress, r1EndAddress, r2StartAddress, r2EndAddress) (LIBMEM_ADDRESS_IN_RANGE(r1StartAddress, r2StartAddress, r2EndAddress) || LIBMEM_ADDRESS_IN_RANGE(r1EndAddress, r2StartAddress, r2EndAddress))


/*! \def LIBMEM_RANGE_OCCLUDES_RANGE(r1StartAddress, r1EndAddress, r2StartAddress, r2EndAddress)
 *
 *  \brief
 *    Macro to determine whether an address range 1 overlaps address range 2 or vice versa.
 *
 *  \param r1StartAddress
 *    The start address of address range 1.
 *
 *  \param r1EndAddress
 *    The end address of address range 1.
 *
 *  \param r2StartAddress
 *    The start address of address range 2.
 *
 *  \param r2EndAddress
 *    The end address of address range 2.
 *
 *  \return
 *    Non-zero if address range 1 overlaps address range 2 or address range 2 overlaps address range 1.
 */
#define LIBMEM_RANGE_OCCLUDES_RANGE(r1StartAddress, r1EndAddress, r2StartAddress, r2EndAddress) (LIBMEM_RANGE_OVERLAPS_RANGE(r1StartAddress, r1EndAddress, r2StartAddress, r2EndAddress) || LIBMEM_RANGE_OVERLAPS_RANGE(r2StartAddress, r2EndAddress, r1StartAddress, r1EndAddress))


/*! \def LIBMEM_ADDRESS_IS_ALIGNED(address, width)
 *
 *  \brief
 *    Macro to determine whether an address is aligned to a specified width.
 *
 *  \param address
 *    The address to check alignment of.
 *
 *  \param width
 *    The alignment width.
 *
 *  \return
 *     Non-zero if address is aligned.
 */
#define LIBMEM_ADDRESS_IS_ALIGNED(address, width) ((((uint32_t)address) & ((width) - 1)) == 0)

/*! \def LIBMEM_ALIGNED_ADDRESS(address, width)
 *
 *  \brief
 *    Macro to return an address aligned to a specified width.
 *
 *  \param address
 *    The address to align.
 *
 *  \param width
 *    The alignment width.
 *
 *  \return
 *    The aligned address.
 */
#define LIBMEM_ALIGNED_ADDRESS(address, width) ((uint8_t *)(((uint32_t)address) & (~((width) - 1))))


/*
 * Type definitions
 */

typedef struct _libmem_driver_handle_t libmem_driver_handle_t;
typedef struct _libmem_driver_functions_t libmem_driver_functions_t;
typedef struct _libmem_ext_driver_functions_t libmem_ext_driver_functions_t;
typedef struct _libmem_geometry_t libmem_geometry_t;
typedef struct _libmem_flash_info_t libmem_flash_info_t;
typedef struct _libmem_sector_info_t libmem_sector_info_t;

typedef int (*libmem_driver_write_fn_t)(libmem_driver_handle_t *h, uint8_t *dest, const uint8_t *src, size_t size);
typedef int (*libmem_driver_fill_fn_t)(libmem_driver_handle_t *h, uint8_t *dest, uint8_t c, size_t size);
typedef int (*libmem_driver_erase_fn_t)(libmem_driver_handle_t *h, uint8_t *start, size_t size, uint8_t **erase_start, size_t *erase_size);
typedef int (*libmem_driver_lock_fn_t)(libmem_driver_handle_t *h, uint8_t *start, size_t size);
typedef int (*libmem_driver_unlock_fn_t)(libmem_driver_handle_t *h, uint8_t *start, size_t size);
typedef int (*libmem_driver_flush_fn_t)(libmem_driver_handle_t *h);

typedef int (*libmem_driver_inrange_fn_t)(libmem_driver_handle_t *h, const uint8_t *dest);
typedef int (*libmem_driver_read_fn_t)(libmem_driver_handle_t *h, uint8_t *dest, const uint8_t *src, size_t size);
typedef uint32_t (*libmem_driver_crc32_fn_t)(libmem_driver_handle_t *h, const uint8_t *start, size_t size, uint32_t crc);

typedef int (*libmem_foreach_driver_fn_t)(libmem_driver_handle_t *h);
typedef int (*libmem_foreach_sector_fn_t)(libmem_driver_handle_t *h, libmem_sector_info_t *sector_info);


/*! \brief
 *    A structure describing a geometry region.
 *
 *  A geometry description can be made up of one or geometry regions. A geometry
 *  region is a collection of equal size sectors.
 */
struct _libmem_geometry_t
{
  unsigned int count; /*!< The number of equal sized sectors in the geometry region. */
  size_t       size;  /*!< The size of the sector. */
};


/*! \brief
 *    A structure containing information about a specific FLASH chip.
 */
struct _libmem_flash_info_t
{
  uint32_t write_timeout_ticks;        /*!< The maximum number of ticks it should take for a write opertion to complete. */
  uint32_t erase_sector_timeout_ticks; /*!< The maximum number of ticks it should take for a sector erase operation to complete. */
  uint32_t erase_chip_timeout_ticks;   /*!< The maximum number of ticks it should take for a chip erase operation to complete. */
  uint32_t max_multi_program_bytes;    /*!< The maximum number of bytes that can be programmed in a multi-program operation. */
  uint16_t primary_cmdset;             /*!< The FLASH chip's primary CFI command set. */
  uint8_t  width;                      /*!< The operating width of the FLASH chip in bytes. */
  uint8_t  pairing;                    /*!< Non-zero if using a paired FLASH configuration. */
};


/*! \brief
 *    A structire describing a sector.
 */
struct _libmem_sector_info_t
{
  int      number; /*!< The sector number (sectors in a geometry are numbered in order from zero). */
  uint8_t *start;  /*!< The start address of the sector. */
  size_t   size;   /*!< The size of the sector. */
};


/*! \brief
 *    A structure containing pointers to a libmem driver's functions.
 */
struct _libmem_driver_functions_t
{
  libmem_driver_write_fn_t   write;       /*!< A pointer to driver's write function. */
  libmem_driver_fill_fn_t    fill;        /*!< A pointer to driver's fill function. */
  libmem_driver_erase_fn_t   erase;       /*!< A pointer to driver's erase function. */
  libmem_driver_lock_fn_t    lock;        /*!< A pointer to driver's lock function. */
  libmem_driver_unlock_fn_t  unlock;      /*!< A pointer to driver's unlock function. */
  libmem_driver_flush_fn_t   flush;       /*!< A pointer to driver's flush function. */
};


/*! \brief
 *    A structure containing pointers to a libmem driver's extended functions.
 */
struct _libmem_ext_driver_functions_t
{
  libmem_driver_inrange_fn_t inrange;     /*!< A pointer to driver's inrange function. */
  libmem_driver_read_fn_t    read;        /*!< A pointer to driver's read function. */
  libmem_driver_crc32_fn_t   crc32;       /*!< A pointer to driver's crc32 function. */
};


/*! \brief
 *    A libmem driver handle.
 */
struct _libmem_driver_handle_t
{
  libmem_driver_handle_t              *next;                 /*!< The next libmem driver in list of drivers. */
  const libmem_driver_functions_t     *driver_functions;     /*!< A pointer to the structure describing the libmem driver's functions. */
  const libmem_ext_driver_functions_t *ext_driver_functions; /*!< A pointer to the structure describing the libmem driver's extended functions. */
  uint8_t                             *start;                /*!< A pointer to the start of the address range handled by the libmem driver. */
  size_t                               size;                 /*!< The size of address range handled by the libmem driver in bytes. */
  const libmem_geometry_t             *geometry;             /*!< A pointer to a null-terminated geometry description list. */
  const libmem_flash_info_t           *flash_info;           /*!< A pointer to the FLASH information structure. */
  uint32_t                             driver_data;          /*!< A data word available for storing driver information. */
  uint32_t                             user_data;            /*!< A data word available for storing user information. */
};


/*! \brief
 *    A global pointer to the first register libmem driver.
 */
extern libmem_driver_handle_t *libmem_drivers;


/*! \brief
 *    Register a libmem driver instance.
 *
 *  \param h
 *    A pointer to the handle of the libmem driver being registered.
 *
 *  \param start
 *    A Pointer to the start of the address range handled by the libmem driver.
 *
 *  \param size
 *    The size of the address range handled by the libmem driver in bytes.
 *
 *  \param geometry
 *    A pointer to al null-terminated geometry description list or NULL if not required.
 *
 *  \param flash_info
 *    A pointer to the FLASH information structure or NULL if not required.
 *
 *  \param driver_functions
 *    A pointer to the structure describing the libmem driver's functions.
 *
 *  \param ext_driver_functions
 *    A pointer to the structure describing the libmem driver's extended functions, or NULL if not required.
 *
 *  This function adds a libmem driver to the list of libmem drivers currently
 *  installed. This function is not normally called directly by an application,
 *  it is typically called by a libmem driver's own register function.
 */
void libmem_register_driver(libmem_driver_handle_t *h, uint8_t *start, size_t size, const libmem_geometry_t *geometry, const libmem_flash_info_t *flash_info, const libmem_driver_functions_t *driver_functions, const libmem_ext_driver_functions_t *ext_driver_functions);


/*! \brief
 *    A helper function for iterating through all sectors handled by a libmem driver.
 *
 *  \param h
 *    A pointer to the handle of the libmem driver.
 *
 *  \param fn
 *    The function to call for each sector.
 *
 *  \return
 *    The libmem status result.
 *
 *  This function iterates through all the sectors handled by a single libmem
 *  driver and calls a \a libmem_foreach_sector_fn_t function for each.
 *
 */
int libmem_foreach_sector(libmem_driver_handle_t *h, libmem_foreach_sector_fn_t fn);


/*! \brief
 *    A helper function for iterating through all sectors handled by a driver that are within a specific address range.
 *
 *  \param h
 *    A pointer to the handle of the libmem driver.
 *
 *  \param range_start
 *    A pointer to the start of the address range.
 *
 *  \param range_size
 *    The size of the address range in bytes.
 *
 *  \param fn
 *    The function to call for each sector.
 *
 *  \param actual_range_start
 *    A pointer to the start of the first sector that is within the address range.
 *
 *  \param actual_range_size
 *    The combined size of all the sectors that are within the address range.
 *
 *  \return
 *    The libmem status result.
 *
 *  This function iterates through all the sectors handled by a single libmem
 *  driver and calls a \a libmem_foreach_sector_fn_t function for each if it
 *  is within the specified address range.
 */
int libmem_foreach_sector_in_range(libmem_driver_handle_t *h, uint8_t *range_start, size_t range_size, libmem_foreach_sector_fn_t fn, uint8_t **actual_range_start, size_t *actual_range_size);


/*! \brief
 *    A helper function that returns the sector number of an address within a specified geometry.
 *
 *  \param start
 *    A pointer to the start address of the geometry desribed by \a geometry.
 *
 *  \param geometry
 *    A pointer to the NULL terminated geometry description.
 *
 *  \param p
 *    A pointer to the address to determine the sector number of.
 *
 *  \return
 *    The sector number or -1 if the address is not located within the described geometry.
 */
int libmem_get_sector_number(uint8_t *start, const libmem_geometry_t *geometry, const uint8_t *p);


/*! \brief
 *   A helper function that returns the number of sectors described by a geometry description.
 *
 *  \param geometry
 *    A pointer to the NULL terminated geometry description.
 *
 *  \return
 *    The number of sectors.
 */
int libmem_get_number_of_sectors(const libmem_geometry_t *geometry);


/*! \brief
 *    A helper function that returns the number of geometry regions described by a geometry description.
 *
 *  \param geometry
 *    A pointer to the NULL terminated geometry description.
 *
 *  \return
 *    The number of geometry regions.
 */
int libmem_get_number_of_regions(const libmem_geometry_t *geometry);


/*! \brief
 *    A helper function that returns the sector information for an address within a specified geometry.
 *
 *  \param start
 *    A pointer to the start address of the geometry desribed by \a geometry.
 *
 *  \param geometry
 *    A pointer to the NULL terminated geometry description.
 *
 *  \param p
 *    A pointer to the address to determine the sector information of.
 *
 *  \param info
 *    A pointer to the \a libmem_sector_info_t structure to write the sector information to.
 *
 *  \return
 *    The libmem status result.
 */
int libmem_get_sector_info(uint8_t *start, const libmem_geometry_t *geometry, const uint8_t *p, libmem_sector_info_t *info);


/*! \brief
 *    A helper function that returns the size of the address range described by a geometry description.
 *
 *  \param geometry
 *    A pointer to the NULL terminated geometry description.
 *
 *  \return
 *    The size of the address range described the by geometry description in bytes.
 */
size_t libmem_get_geometry_size(const libmem_geometry_t *geometry);


/*! \brief 
 *    Compute CRC-32 checksum of an address range
 *
 *  \param start
 *    A pointer to the start of the address range.
 *
 *  \param size
 *    The size of the address range in bytes.
 *
 *  \param crc
 *    The initial CRC-32 value.
 *
 *  \return
 *    The computed CRC-32 value.
 *
 *  This function computes a CRC-32 checksum on a block of data using the
 *  standard CRC-32 polynomial (0x04C11DB7). Note that this implementation
 *  doesn't reflect the input or the output and the result is not inverted.
 *
 *  Example:
 *  \code
 *    uint32_t crc = 0xFFFFFFFF;
 *
 *    crc = libmem_crc32_direct((uint8_t *)0x10000000, 1024, crc);
 *
 *  \endcode
 */
uint32_t libmem_crc32_direct(const uint8_t *start, size_t size, uint32_t crc);


/*! \brief
 *    Write a block of data using a libmem driver.
 *
 *  \param dest
 *    A pointer to the address to write the block of data.
 *
 *  \param src
 *    A pointer to the address to copy the block of data from.
 *
 *  \param size
 *    The size of the block of data to copy in bytes.
 *
 *  \return
 *    The libmem status result.
 *
 *  This function locates the libmem driver for the address pointed to
 *  by \a start and then calls the libmem driver's \a write function.
 *
 *  Note that the address range being written to cannot span multiple
 *  libmem drivers.
 *
 *  Example:
 *  \code
 *    const unsigned char buffer[8] = { 1, 2, 3, 4, 5, 6, 7, 8 };
 *    int res;
 *
 *    res = libmem_write((uint8_t *)0x10000000, buffer, sizeof(buffer));
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_write : success\n")
 *    else
 *      printf("libmem_write : failed (%d)\n", res);
 *  \endcode
 */
int libmem_write(uint8_t *dest, const uint8_t *src, size_t size);


/*! \brief
 *    Fill memory with a specific data value using a libmem driver.
 *
 *  \param dest
 *    A pointer to the address to write the data.
 *
 *  \param c
 *    The data value to fill the memory with.
 *
 *  \param size
 *    The number of bytes to write.
 *
 *  \return
 *    The libmem status result.
 *
 *  This function locates the libmem driver for the address pointed to by \a dest
 *  and then calls the libmem driver's \a fill function.
 *
 *  Note that the address range being written to cannot span multiple
 *  libmem drivers.
 *
 *  Example:
 *  \code
 *    int res;
 *
 *    res = libmem_fill((uint8_t *)0x10000000, 0xCC, 64);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_fill : success\n")
 *    else
 *      printf("libmem_fill : failed (%d)\n", res);
 * 
 *  \endcode
 */
int libmem_fill(uint8_t *dest, uint8_t c, size_t size);


/*! \brief
 *    Read a block of data using a libmem driver.
 *
 *  \param dest
 *    A pointer to the address to write the block of data.
 *
 *  \param src
 *    A pointer to the address to copy the block of data from.
 *
 *  \param size 
 *    The size of the block of data to copy in bytes.
 *
 *  \return
 *    The libmem status result.
 *
 *  This function locates the libmem driver for the address pointed to
 *  by \a src and then calls the libmem driver's \a read extended function if
 *  it has been implemented. If the \a read function has not been implemented
 *  then the memory will be read directly using memcpy.
 *  The intention for this function is to allow you to use the libmem
 *  library for memory that doesn't appear on the address bus by providing
 *  a virtual address range for the device.
 *
 *  Note that if the libmem driver's read function is used, the address range 
 *  being read cannot span multiple libmem drivers.
 *
 *  Example:
 *  \code
 *
 *    uint8_t buffer[64];
 *    int res;
 *
 *    res = libmem_read(buffer, (uint8_t *)0x10000000, sizeof(buffer));
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_read : success\n")
 *    else
 *      printf("libmem_read : failed (%d)\n", res);
 *   
 *  \endcode
 */
int libmem_read(uint8_t *dest, const uint8_t *src, size_t size);


/*! \brief
 *    Erase a block of memory using a libmem driver.
 *
 *  \param start 
 *    A pointer to the start address of the memory range to erase.
 *
 *  \param size 
 *    The size of the memory range to erase in bytes.
 *
 *  \param erase_start 
 *    A pointer to a location in memory to store a pointer to the start of
 *    the memory range that has actually been erased or NULL if not required.
 *
 *  \param erase_size
 *    A pointer to a location in memory to store the size in bytes of the
 *    memory range that has actually been erased or NULL if not required.
 *
 *  \return 
 *    The libmem status result.
 *
 *  This function locates the libmem driver for the address pointed to 
 *  by \a start and then calls the libmem driver's \a erase function.
 *
 *  Note that the address range being erased cannot span multiple libmem
 *  drivers.
 *
 *  Example:
 *  \code
 *
 *    uint8_t *erase_start;
 *    size_t erase_size;
 *    int res;
 *
 *    res = libmem_erase((uint8_t *)0x10000000, 1024, &erase_start, &erase_size);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_erase : success (erased %08X - 0x%08X)\n", erase_start, erase_start + erase_size - 1);
 *    else
 *      printf("libmem_erase : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_erase(uint8_t *start, size_t size, uint8_t **erase_start, size_t *erase_size);


/*! \brief
 *   Erase all memory using libmem drivers.
 *
 *  \return 
 *    The libmem status result.
 *
 *  This function iterates through all registered libmem drivers calling each
 *  driver's \a erase function specifying the drivers entire memory range as its
 *  parameters.
 *
 *  The function will terminate if any of the driver's \a erase functions return
 *  a result other than \a LIBMEM_STATUS_SUCCESS.
 *
 *  Example:
 *  \code
 *
 *    int res;
 *
 *    res = libmem_erase_all();
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_erase_all : success\n");
 *    else
 *      printf("libmem_erase_all : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_erase_all(void);


/*! \brief
 *    Lock a block of memory using a libmem driver.
 *
 *  \param start 
 *    A pointer to the start address of the memory range to lock.
 *
 *  \param size 
 *    The size of the memory range to lock in bytes.
 *
 *  \return 
 *    The libmem status result.
 *
 *  This function locates the libmem driver for the address pointed to 
 *  by \a start and then calls the libmem driver's \a lock function.
 *
 *  Example:
 *  \code
 *    int res;
 *
 *    res = libmem_lock((uint8_t *)0x10000000, 1024);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_lock : success\n");
 *    else
 *      printf("libmem_lock : failed (%d)\n", res);
 *  \endcode
 */
int libmem_lock(uint8_t *start, size_t size);


/*! \brief
 *   Lock all memory using libmem drivers.
 *
 *  \return
 *    The libmem status result.
 *
 *  This function iterates through all registered libmem drivers calling each
 *  driver's \a lock function specifying the drivers entire memory range as its
 *  parameters.
 *
 *  The function will terminate if any of the driver's \a lock functions return
 *  a result other than \a LIBMEM_STATUS_SUCCESS.
 *
 *  Example:
 *  \code
 *
 *    int res;
 *
 *    res = libmem_lock_all();
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_lock_all : success\n");
 *    else
 *      printf("libmem_lock_all : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_lock_all(void);


/*! \brief
 *    Unlock a block of memory using a libmem driver.
 *
 *  \param start
 *    A pointer to the start address of the memory range to unlock.
 *
 *  \param size
 *    The size of the memory range to unlock in bytes.
 *
 *  \return
 *    The libmem status result.
 *
 *  This function locates the libmem driver for the address pointed to
 *  by \a start and then calls the libmem driver's \a unlock function.
 *
 *  Example:
 *  \code
 *    int res;
 *
 *    res = libmem_unlock((uint8_t *)0x10000000, 1024);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_unlock : success\n");
 *    else
 *      printf("libmem_unlock : failed (%d)\n", res);
 *  \endcode
 */
int libmem_unlock(uint8_t *start, size_t size);


/*! \brief
 *   Unlock all memory using libmem drivers.
 *
 *  \return
 *    The libmem status result.
 *
 *  This function iterates through all registered libmem drivers calling each
 *  driver's \a unlock function specifying the drivers entire memory range as its
 *  parameters.
 *
 *  The function will terminate if any of the driver's \a unlock functions return
 *  a result other than \a LIBMEM_STATUS_SUCCESS.
 *
 *  Example:
 *  \code
 *
 *    int res;
 *
 *    res = libmem_unlock_all();
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_unlock_all : success\n");
 *    else
 *      printf("libmem_unlock_all : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_unlock_all(void);


/*! \brief
 *    Flush any outstanding memory operations and return memory to read mode if applicable.
 *
 *  \return
 *    The libmem status result.
 *
 *  libmem drivers do not necessarily carry out operations immediately or they may 
 *  leave the memory in an unreadable state for performance reasons. You should
 *  call \a libmem_flush once you have finished carrying out memory operations in order
 *  to complete all outstanding transactions and return the memory to a readable
 *  state.
 *
 *  Example:
 *  \code
 *
 *    int res;
 *
 *    res = libmem_flush();
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_flush : success\n");
 *    else
 *      printf("libmem_flush : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_flush(void);


/*! \brief
 *    Compute CRC-32 checksum of an address range using a libmem driver.
 *
 *  \param start
 *    A pointer to the start of the address range.
 *
 *  \param size
 *    The size of the address range in bytes.
 *
 *  \param crc
 *    The initial CRC-32 value.
 *
 *  \return
 *    The computed CRC-32 value.
 *
 *  This function locates the libmem driver for the address pointed to
 *  by \a start and then calls the libmem driver's \a crc32 extended 
 *  function if it has one and returns the result. If the driver
 *  hasn't implemented the \a crc32 extended function then the
 *  \a libmem_crc32_direct is called which accesses the memory directly.
 *  The intention for this function is to allow you to use the libmem
 *  library for memory that doesn't appear on the address bus by providing
 *  a virtual address range for the device.
 *
 *  Example:
 *  \code
 *    uint32_t crc = 0xFFFFFFFF;
 *
 *    crc = libmem_crc32((uint8_t *)0x10000000, 1024, crc);
 *
 *  \endcode
 *   
 */
uint32_t libmem_crc32(const uint8_t *start, size_t size, uint32_t crc);


#ifdef LIBMEM_LIGHT

#define libmem_get_driver(p1) libmem_drivers
#define libmem_foreach_driver(p1) p1(libmem_drivers)
#define LIBMEM_INLINE
#else

libmem_driver_handle_t *libmem_get_driver(const uint8_t *dest);
int libmem_foreach_driver(libmem_foreach_driver_fn_t fn);
#define LIBMEM_INLINE inline

#endif


/*! \brief
 *    Return a FLASH memory device's common flash interface (CFI) information.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \param size
 *    A pointer to the memory location to store the size (in bytes) of the FLASH memory.
 *
 *  \param geometry
 *    A pointer to the memory location to store the geometry description or NULL if not
 *    required.
 *
 *  \param max_geometry_regions
 *    The maximum number of geometry regions that can be stored at the memory pointed
 *    to by \a geometry. The geometry description is NULL terminated so 
 *    \a max_geometry_regions must be at least two regions in size in order to store one
 *    geometry region and one terminator entry.
 *
 *  \param flash_info
 *    A pointer to the memory location to store the remaining FLASH information, or NULL
 *    if not required.
 *
 *  \return
 *    The libmem status result.
 * 
 *  This function attempts to return the FLASH device's type, size, geometry and other
 *  FLASH information from only a pointer to the first address the FLASH memory is
 *  located at. It uses the common flash memory interface (CFI) to obtain this
 *  information and therefore only works on FLASH devices that fully support this
 *  interface.
 *
 *  Example:
 *  \code
 *
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    libmem_flash_info_t flash1_info;
 *    const int flash1_max_geometry_regions = 4;
 *    libmem_geometry_t flash1_geometry[flash1_max_geometry_regions];
 *    size_t flash1_size;
 *    int res;
 *
 *    res = libmem_cfi_get_info(flash1_start,
 *                              &flash1_size,
 *                              flash1_geometry,
 *                              flash1_max_geometry_regions,
 *                              &flash1_info);
 * 
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_cfi_get_info : success\n");
 *    else
 *      printf("libmem_cfi_get_info : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_cfi_get_info(uint8_t *start, size_t *size, libmem_geometry_t *geometry, int max_geometry_regions, libmem_flash_info_t *flash_info);


/*! \brief
 *    Register a libmem FLASH driver based on the FLASH's CFI information.
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \param geometry
 *    A pointer to the memory location to store the geometry description.
 *
 *  \param max_geometry_regions
 *    The maximum number of geometry regions that can be stored at the memory pointed
 *    to by \a geometry. The geometry description is NULL terminated so 
 *    \a max_geometry_regions must be at least two regions in size in order to store one
 *    geometry region and one terminator entry.
 *
 *  \param flash_info
 *    A pointer to the memory location to store the remaining FLASH information.
 *
 *  \return
 *    The libmem status result.
 *
 *  This function calls \a libmem_cfi_get_info to detect the type and geometry of the
 *  the FLASH pointed to by \a start and then, if the FLASH memory is supported, 
 *  registers an appropriate libmem driver for the FLASH.
 *
 *  Use of this function requires all supported CFI libmem drivers to be linked in,
 *  therefore if memory is at a premium you should register only the libmem
 *  FLASH driver you require instead of using this function.
 *
 *  Example:
 *  \code
 *
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    libmem_flash_info_t flash1_info;
 *    const int flash1_max_geometry_regions = 4;
 *    libmem_geometry_t flash1_geometry[flash1_max_geometry_regions];
 *    libmem_driver_handle_t flash1_handle;
 *    int res;
 *
 *    res = libmem_register_cfi_driver(&flash1_handle,
 *                                     flash1_start,
 *                                     flash1_geometry,
 *                                     flash1_max_geometry_regions,
 *                                     &flash1_info);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_register_cfi_driver : success\n");
 *    else
 *      printf("libmem_register_cfi_driver : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_register_cfi_driver(libmem_driver_handle_t *h, uint8_t *start, libmem_geometry_t *geometry, int max_geometry_regions, libmem_flash_info_t *flash_info);

/*! \brief
 *    Register an 8 bit CFI command set 1 (Intel Extended) libmem driver
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \param size
 *    The size of the FLASH memory.
 *
 *  \param geometry
 *    A NULL terminated description of the FLASH's geometry.
 *
 *  \param flash_info
 *    A pointer to the FLASH information structure or NULL if not required.
 *
 *  \return
 *    The libmem status result.
 *
 *  Example:
 *  \code
 *  
 *    libmem_driver_handle_t flash1_handle;
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    libmem_geometry_t flash1_geometry[] =
 *      {
 *        { 8,  0x00002000 }, // 8 x 8KB sectors
 *        { 31, 0x00010000 }, // 31 x 64KB sectors
 *        { 0, 0 },           // NULL terminator
 *      };
 *    int res;
 *
 *    res = libmem_register_cfi_0001_8_driver(&flash1_handle,
 *                                            flash1_start,
 *                                            libmem_get_geometry_size(flash1_geometry),
 *                                            flash1_geometry,
 *                                            0);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_register_cfi_0001_8_driver : success\n");
 *    else
 *      printf("libmem_register_cfi_0001_8_driver : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_register_cfi_0001_8_driver(libmem_driver_handle_t *h, uint8_t *start, size_t size, const libmem_geometry_t *geometry, const libmem_flash_info_t *flash_info);

/*! \brief
 *    Register a 16 bit CFI command set 1 (Intel Extended) libmem driver
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \param size
 *    The size of the FLASH memory.
 *
 *  \param geometry
 *    A NULL terminated description of the FLASH's geometry.
 *
 *  \param flash_info
 *    A pointer to the FLASH information structure or NULL if not required.
 *
 *  \return
 *    The libmem status result.
 *
 *  Example:
 *  \code
 *
 *    libmem_driver_handle_t flash1_handle;
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    libmem_geometry_t flash1_geometry[] =
 *      {
 *        { 8,  0x00002000 }, // 8 x 8KB sectors
 *        { 31, 0x00010000 }, // 31 x 64KB sectors
 *        { 0, 0 },           // NULL terminator
 *      };
 *    int res;
 *
 *    res = libmem_register_cfi_0001_16_driver(&flash1_handle,
 *                                             flash1_start,
 *                                             libmem_get_geometry_size(flash1_geometry),
 *                                             flash1_geometry,
 *                                             0);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_register_cfi_0001_16_driver : success\n");
 *    else
 *      printf("libmem_register_cfi_0001_16_driver : failed (%d)\n", res);
 *  
 *  \endcode
 */           
int libmem_register_cfi_0001_16_driver(libmem_driver_handle_t *h, uint8_t *start, size_t size, const libmem_geometry_t *geometry, const libmem_flash_info_t *flash_info);


/*! \brief
 *    Register a 16 bit CFI command set 2 (AMD Standard) libmem driver
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \param size
 *    The size of the FLASH memory.
 *
 *  \param geometry
 *    A NULL terminated description of the FLASH's geometry.
 *
 *  \param flash_info
 *    A pointer to the FLASH information structure or NULL if not required.
 *
 *  \return
 *    The libmem status result.
 *
 *  Example:
 *  \code
 *
 *    libmem_driver_handle_t flash1_handle;
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    libmem_geometry_t flash1_geometry[] =
 *      {
 *        { 8,  0x00002000 }, // 8 x 8KB sectors
 *        { 31, 0x00010000 }, // 31 x 64KB sectors
 *        { 0, 0 },           // NULL terminator
 *      };
 *    int res;
 *
 *    res = libmem_register_cfi_0002_16_driver(&flash1_handle,
 *                                             flash1_start,
 *                                             libmem_get_geometry_size(flash1_geometry),
 *                                             flash1_geometry,
 *                                             0);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_register_cfi_0002_16_driver : success\n");
 *    else
 *      printf("libmem_register_cfi_0002_16_driver : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_register_cfi_0002_16_driver(libmem_driver_handle_t *h, uint8_t *start, size_t size, const libmem_geometry_t *geometry, const libmem_flash_info_t *flash_info);


/*! \brief
 *    Register an 8 bit CFI command set 3 (Intel Standard) libmem driver
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \param size
 *    The size of the FLASH memory.
 *
 *  \param geometry
 *    A NULL terminated description of the FLASH's geometry.
 *
 *  \param flash_info
 *    A pointer to the FLASH information structure or NULL if not required.
 *
 *  \return
 *    The libmem status result.
 *
 *  Example:
 *  \code
 *
 *    libmem_driver_handle_t flash1_handle;
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    libmem_geometry_t flash1_geometry[] =
 *      {
 *        { 8,  0x00002000 }, // 8 x 8KB sectors
 *        { 31, 0x00010000 }, // 31 x 64KB sectors
 *        { 0, 0 },           // NULL terminator
 *      };
 *    int res;
 *
 *    res = libmem_register_cfi_0003_8_driver(&flash1_handle,
 *                                            flash1_start,
 *                                            libmem_get_geometry_size(flash1_geometry),
 *                                            flash1_geometry,
 *                                            0);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_register_cfi_0003_8_driver : success\n");
 *    else
 *      printf("libmem_register_cfi_0003_8_driver : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_register_cfi_0003_8_driver(libmem_driver_handle_t *h, uint8_t *start, size_t size, const libmem_geometry_t *geometry, const libmem_flash_info_t *flash_info);


/*! \brief
 *    Register an 16 bit CFI command set 3 (Intel Standard) libmem driver
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \param size
 *    The size of the FLASH memory.
 *
 *  \param geometry
 *    A NULL terminated description of the FLASH's geometry.
 *
 *  \param flash_info
 *    A pointer to the FLASH information structure or NULL if not required.
 *
 *  \return
 *    The libmem status result.
 *
 *  Example:
 *  \code
 *
 *    libmem_driver_handle_t flash1_handle;
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    libmem_geometry_t flash1_geometry[] =
 *      {
 *        { 8,  0x00002000 }, // 8 x 8KB sectors
 *        { 31, 0x00010000 }, // 31 x 64KB sectors
 *        { 0, 0 },           // NULL terminator
 *      };
 *    int res;
 *
 *    res = libmem_register_cfi_0003_16_driver(&flash1_handle,
 *                                             flash1_start,
 *                                             libmem_get_geometry_size(flash1_geometry),
 *                                             flash1_geometry,
 *                                             0);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_register_cfi_0003_16_driver : success\n");
 *    else
 *      printf("libmem_register_cfi_0003_16_driver : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_register_cfi_0003_16_driver(libmem_driver_handle_t *h, uint8_t *start, size_t size, const libmem_geometry_t *geometry, const libmem_flash_info_t *flash_info);


/*! \brief
 *    Register a combined multi-width CFI command set 1 & 3 (Intel) libmem driver
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \param size
 *    The size of the FLASH memory.
 *
 *  \param geometry
 *    A NULL terminated description of the FLASH's geometry.
 *
 *  \param flash_info
 *    A pointer to the FLASH information structure.
 *
 *  \return
 *    The libmem status result.
 *
 *  This function registers a combined multi-width CFI command set 1 & 3 (Intel)
 *  libmem driver. The advantage of this driver over the individual single
 *  width and command set drivers is that one driver will support a range
 *  of Intel FLASH chips, the disadvantage is that of increased code size
 *  and reduced performance.
 *
 *  Example:
 *  \code
 *
 *    const int flash1_max_geometry_regions = 4;
 *    libmem_driver_handle_t flash1_handle;
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    libmem_geometry_t flash1_geometry[flash1_max_geometry_regions];
 *    libmem_flash_info_t flash1_info;
 *    size_t flash1_size;
 *    int res;
 *
 *    // Detect the type, size and geometry of the Intel FLASH.
 *    res = libmem_cfi_get_info(flash1_start,
 *                              &flash1_size,
 *                              flash1_geometry,
 *                              flash1_max_geometry_regions,
 *                              &flash1_info);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      {
 *        libmem_register_cfi_intel_driver(&flash1_handle, flash1_start, flash1_size, flash1_geometry, &flash1_info);
 *
 *        // Register the driver
 *        res = libmem_register_cfi_intel_driver(&flash1_handle,
 *                                               flash1_start,
 *                                               flash1_size),
 *                                               flash1_geometry,
 *                                               flash1_info);
 *
 *        if (res == LIBMEM_STATUS_SUCCESS)
 *          printf("libmem_register_cfi_intel_driver : success\n");
 *        else
 *          printf("libmem_register_cfi_intel_driver : failed (%d)\n", res);
 *
 *      }
 *    else
 *      printf("libmem_cfi_get_info : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_register_cfi_intel_driver(libmem_driver_handle_t *h, uint8_t *start, size_t size, const libmem_geometry_t *geometry, const libmem_flash_info_t *flash_info);


/*! \brief
 *    Register an libmem driver for an ST M28W320CB FLASH chip.
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \return
 *    The libmem status result.
 *
 *  Example:
 *  \code
 *
 *    libmem_driver_handle_t flash1_handle;
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    int res;
 *
 *    res = libmem_register_st_m28w320cb_driver(&flash1_handle, flash1_start);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_register_st_m28w320cb_driver : success\n");
 *    else
 *      printf("libmem_register_st_m28w320cb_driver : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_register_st_m28w320cb_driver(libmem_driver_handle_t *h, uint8_t *start);


/*! \brief
 *    Register an libmem driver for an ST M28W320CT FLASH chip.
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \return
 *    The libmem status result.
 *
 *  Example:
 *  \code
 *
 *    libmem_driver_handle_t flash1_handle;
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    int res;
 *
 *    res = libmem_register_st_m28w320ct_driver(&flash1_handle, flash1_start);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_register_st_m28w320ct_driver : success\n");
 *    else
 *      printf("libmem_register_st_m28w320ct_driver : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_register_st_m28w320ct_driver(libmem_driver_handle_t *h, uint8_t *start);


/*! \brief
 *    Register a libmem driver for a 16 bit SST39xFx00A FLASH chip.
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the FLASH memory.
 *
 *  \param size
 *    The size of the FLASH memory.
 *
 *  \param geometry
 *    A NULL terminated description of the FLASH's geometry.
 *
 *  \param flash_info
 *    A pointer to the FLASH information structure or NULL if not required.
 *
 *  \return
 *    The libmem status result.
 *
 *  Example:
 *  \code
 *
 *    libmem_driver_handle_t flash1_handle;
 *    uint8_t *flash1_start = (uint8_t *)0x10000000;
 *    libmem_flash_info_t flash1_info;
 *    const int flash1_max_geometry_regions = 4;
 *    libmem_geometry_t flash1_geometry[flash1_max_geometry_regions];
 *    size_t size;
 *    int res;
 *   
 *    // Get CFI FLASH information and geometry 
 *    res = libmem_cfi_get_info(flash1_start, &size, flash1_geometry, flash1_max_geometry_regions, &flash1_info);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      {
 *        res = libmem_register_sst39xFx00A_16_driver(&flash1_handle, flash1_start, size, flash1_geometry, &flash1_info)
 *
 *        if (res == LIBMEM_STATUS_SUCCESS)
 *          printf("libmem_register_sst39xFx00A_16_driver : success\n");
 *        else
 *          printf("libmem_register_sst39xFx00A_16_driver : failed (%d)\n", res); 
 *      }
 *    else
 *      printf("libmem_cfi_get_info : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_register_sst39xFx00A_16_driver(libmem_driver_handle_t *h, uint8_t *start, size_t size, const libmem_geometry_t *geometry, const libmem_flash_info_t *flash_info);

/*! \brief
 *    Register a simple libmem driver that directly accesses RAM.
 *
 *  \param h
 *    A pointer to the libmem handle structure to use for this libmem driver.
 *
 *  \param start
 *    The start address of the RAM.
 *
 *  \param size
 *    The size of the RAM.
 *
 *  \return
 *    The libmem status result.
 *
 *  Example:
 *  \code
 *
 *    libmem_driver_handle_t ram1_handle;
 *    uint8_t *ram1_start = (uint8_t *)0x10000000;
 *    const size_t ram1_size = 1024;
 *    int res;
 *
 *    res = libmem_register_ram_driver(&ram_handle, ram1_start, ram1_size);
 *
 *    if (res == LIBMEM_STATUS_SUCCESS)
 *      printf("libmem_register_ram_driver : success\n");
 *    else
 *      printf("libmem_register_ram_driver : failed (%d)\n", res);
 *
 *  \endcode
 */
int libmem_register_ram_driver(libmem_driver_handle_t *h, uint8_t *start, size_t size);

#ifdef __cplusplus
}
#endif

#endif

