// ARM Runtime Support.
//
// Copyright (c) 2006 Rowley Associates Limited.
//
// This file may be distributed under the terms of the License Agreement
// provided with this software.
//
// THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#ifndef libarm_h
#define libarm_h

#include <stdlib.h>


/*! \brief Enables or disables IRQ interrupts.
 *
 *  \param enable  If non-zero IRQ interrupts will be enabled, otherwise they
 *                 will be disabled. 
 */
void libarm_set_irq(int enable);


/*! \brief Enables or disables FIQ interrupts.
 *
 *  \param enable  If non-zero FIQ interrupts will be enabled, otherwise they
 *                 will be disabled.
 */
void libarm_set_fiq(int enable);


/*! \brief Disables IRQ and FIQ interrupts and return the previous enable state.
 *
 *  \return The IRQ and FIQ enable state prior to disabling the IRQ and FIQ
 *          interrupts.
 */
int libarm_disable_irq_fiq(void);


/*! \brief Restores the IRQ and FIQ interrupt enable state to the state prior
 *         to calling \a libarm_disable_irq_fiq.
 *
 *  \param disable_irq_fiq_return  The value returned from \a
 *                                 libarm_disable_irq_fiq.
 */
void libarm_restore_irq_fiq(int disable_irq_fiq_return);


/*! \brief Re-enable ARM's global interrupts from within an IRQ ISR.
 *
 *  This call must be accompanied with a call to \a libarm_isr_disable_irq 
 *  prior to completion of the ISR.
 */
void libarm_isr_enable_irq(void);


/*! \brief Re-disable ARM's global interrupts from within an IRQ ISR.
 *
 *  A call to \a libarm_isr_enable_irq must have been made prior to calling
 *  this function.
 */
void libarm_isr_disable_irq(void);


/*! \brief Write a word of data to the host over JTAG using the ARM's debug
 *         comms channel.
 *
 *  The ARM's debug comms channel is usually used by debuggers so writing to
 *  this port with a debugger attached can cause unpredictable results.
 *
 *  \param data  The data to write to the debug comms channel.
 */
void libarm_dcc_write(unsigned long data);


/*! \brief Read a word of data from the host over JTAG using the ARM's debug
 *         comms channel.
 *
 *  The ARM's debug comms channel is usually used by debuggers so writing to
 *  this port with a debugger attached can cause unpredictable results.
 *
 *  \return The data read from the debug comms channel.
 */
unsigned long libarm_dcc_read(void);


/*! \brief Serve commands from ARM's communication channel until terminated.
 */
void libarm_run_dcc_port_server(void);


/*! \brief Create a flat mapped (i.e. virtual addresses == physical addresses)
 *         level 1 translation table.
 *
 *  \param translation_table  Pointer to the start of the translation table.
 */
void libarm_mmu_flat_initialise_level_1_table(void *translation_table);


/*! \brief Create a level 2 small page table for an address range.
 *
 *  This function requires a level 1 translation table to be created using 
 *  \a libarm_mmu_flat_initialise_level_1_table prior to calling.
 *
 *  \param translation_table   A pointer to the start of the translation table.
 *
 *  \param start               A pointer to the start address of the address
 *                             range.
 *
 *  \param size                The size of the address range in bytes.
 *
 *  \param coarse_page_tables  A pointer to the start address of the coarse page
 *                             tables.
 */

void libarm_mmu_flat_initialise_level_2_small_page_table(void *translation_table, void *start, size_t size, void *coarse_page_tables);


/*! \brief Mark region of memory described by level 1 section descriptors as
 *         cacheable.
 *
 *  This function requires a level 1 translation table to be created using
 *  \a libarm_mmu_flat_initialise_level_1_table prior to calling.
 * 
 *  \param translation_table  A pointer to the start of the translation table.
 *
 *  \param start              A pointer to the start of the cacheable region.
 *
 *  \param size               The size of the cacheable region in bytes.
 */

void libarm_mmu_flat_set_level_1_cacheable_region(void *translation_table, void *start, size_t size);


/*! \brief Mark region of memory described by level 2 small page table
 *         descriptors as cacheable.
 *
 *  This function requires a level 2 small page table table to be created using
 *  \a libarm_mmu_flat_initialise_level_2_small_page_table prior to calling.
 *
 *  \param translation_table  A pointer to the start of the translation table.
 *
 *  \param start              A pointer to the start address of the cacheable
 *                            region.
 *
 *  \param size               The size of the cacheable region in bytes.
 */

void libarm_mmu_flat_set_level_2_small_page_cacheable_region(void *translation_table, void *start, size_t size);

#endif
