// CrossWorks Tasking Library.
//
// Copyright (c) 2004 Rowley Associates Limited.
//
// This file may be distributed under the terms of the License Agreement
// provided with this software.
//
// THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#include <targets/PXA250.h>
#include "ctl_api.h"

#define INTERRUPT_SOURCE_COUNT 32

static CTL_ISR_FN_t vectors[INTERRUPT_SOURCE_COUNT];

static void
irq_handler2(void)
{
  unsigned int status = ICIP;
  int i;
  for (i = 0; i < INTERRUPT_SOURCE_COUNT; ++i)
    {
      if (status & 1)
        {
          ICMR &= ~(1 << i);
          vectors[i]();
          ICMR |= 1 << i;
          break;
        }
      status >>= 1;
    }
}

void irq_handler(void) __attribute__((naked));

void
irq_handler(void)
{
#ifdef CTL_TASKING
  asm("stmfd sp!, {r0-r12, lr}");
  asm("mrs r0, spsr");
  asm("stmfd sp!, {r0}");
  ctl_interrupt_count++;
#else
  // store the APCS registers in the non-tasking case
  asm("stmfd sp!, {r0-r4, r12, lr}");
#endif
  irq_handler2();
#ifdef CTL_TASKING
  asm("mov r0, sp");
  asm("ldr r1, =ctl_exit_isr");
  asm("bx r1");
#else
  // return from interrupt
  asm("ldmfd sp!, {r0-r4, r12, lr}");
  asm("subs pc, lr, #4");
#endif
}

int
ctl_set_isr(unsigned int vector, unsigned int priority, CTL_ISR_TRIGGER_t trigger, CTL_ISR_FN_t isr, CTL_ISR_FN_t *oldisr)
{
  if (oldisr)
    *oldisr = vectors[vector];
  vectors[vector] = isr;
  return 1;
}

int
ctl_unmask_isr(unsigned int vector)
{
  ICMR |= 1 << vector;
  return 1;
}

int
ctl_mask_isr(unsigned int vector)
{
  ICMR &= ~(1 << vector);
  return 1;
}

#define TIMER0_INT 26

#define CORE_CLOCK_FREQUENCY 3686400
#define TIMER_CORE_CLOCK_DIVIDER 1
#define TIMER_FREQUENCY 1000
#define TIMER_RECHARGE (CORE_CLOCK_FREQUENCY / TIMER_CORE_CLOCK_DIVIDER / TIMER_FREQUENCY)

static CTL_ISR_FN_t userTimerISR;

static void
timerISR(void)
{
  OSSR = 1; /* Clear the timer interrupt */
  OSMR0 = OSCR + TIMER_RECHARGE;
  userTimerISR();
}

void
ctl_start_timer(CTL_ISR_FN_t isr)
{
  ICLR &= ~(1 << TIMER0_INT); /* Timer interrupt generates IRQ interrupts */
  ICMR |= 1 << TIMER0_INT; /* Unmask the timer interrupt */
  OSMR0 = OSCR + TIMER_RECHARGE;
  OIER |= 1; /* Enable OSMR0 match interrupt */
  userTimerISR = isr;
  ctl_set_isr(TIMER0_INT, 1, CTL_ISR_TRIGGER_LOW_LEVEL, timerISR, 0);
  ctl_unmask_isr(TIMER0_INT);
}

unsigned long
ctl_get_ticks_per_second(void)
{
  return 1000;
}      
