// CrossWorks Tasking Library.
//
// Copyright (c) 2004 Rowley Associates Limited.
//
// This file may be distributed under the terms of the License Agreement
// provided with this software.
//
// THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#include <targets/TMS470.h>
#include "ctl_api.h"

#define INTERRUPT_SOURCE_COUNT 32

static CTL_ISR_FN_t vectors[INTERRUPT_SOURCE_COUNT];

void irq_handler(void) __attribute__((naked));

void
irq_handler(void)
{
  // check if I bit is set in SPSR to detect spurious interrupts
  asm("stmfd sp!, {r0}");
  asm("mrs r0, spsr");
  asm("tst r0, #0x80");
  asm("ldmfd sp!, {r0}");
  asm("subnes pc, lr, #4");
#ifdef CTL_TASKING
  asm("stmfd sp!, {r0-r12, lr}"); 
  asm("mrs r0, spsr");
  asm("stmfd sp!, {r0}");
  ctl_interrupt_count++;  
#else
  // store the APCS registers in the non-tasking case
  asm("stmfd sp!, {r0-r4, r12, lr}");
#endif  
  vectors[(0xFF & IRQIVEC) - 1]();
#ifdef CTL_TASKING
  asm("mov r0, sp");
  asm("ldr r1, =ctl_exit_isr");
  asm("bx r1");
#else
  // return from interrupt
  asm("ldmfd sp!, {r0-r4, r12, lr}");
  asm("subs pc, lr, #4");
#endif
}

int
ctl_set_isr(unsigned int vector, unsigned int priority, CTL_ISR_TRIGGER_t trigger, CTL_ISR_FN_t isr, CTL_ISR_FN_t *oldisr)
{
  if (oldisr)
    *oldisr = vectors[vector];
  vectors[vector] = isr;
  return 1;
}

int
ctl_unmask_isr(unsigned int vector)
{
  REQMASK |= 1 << vector;
  return 1;
}

int
ctl_mask_isr(unsigned int vector)
{
  REQMASK &= ~(1 << vector);
  return 1;
}

#define RTI_CMP2_INT 1
#define RTI_CMP1_INT 2

static CTL_ISR_FN_t userTimerISR;

static void
timerISR(void)
{
  RTICINT &= ~0x80; /* Clear CM1 flag */
  RTICMP1 += 1000; /* Setup the next CMP1 interrupt */
  userTimerISR();
}

void
ctl_start_timer(CTL_ISR_FN_t isr)
{
  RTIPCTL = 11; /* Clock timer counter at 1Mhz */
  RTICINT |= 0x20; /* Enable CMP1 interrupt */
  RTICMP1 = 1000; /* Setup initial CMP1 interrupt */
  RTICNTR &= 0x0000FFFF; /* Reset the counter */
  userTimerISR = isr;
  ctl_set_isr(RTI_CMP1_INT, 0, CTL_ISR_TRIGGER_FIXED, timerISR, 0);
  ctl_unmask_isr(RTI_CMP1_INT);
}

unsigned long
ctl_get_ticks_per_second()
{
  return 1000;
}
