// STMicroelectronics STR73x FLASH driver.
//
// Copyright (c) 2005 Rowley Associates Limited.
//
// This file may be distributed under the terms of the License Agreement
// provided with this software.
//
// THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#include <libmem.h>

typedef volatile unsigned long *flash_pointer_t;

#define FLASH_START_ADDRESS 0x80000000
#define FLASH_REG(offset) *((volatile unsigned int *)(FLASH_START_ADDRESS | offset))

#define FCR0_OFFSET 0x100000 /* Flash Control Register 0 */
#define FCR1_OFFSET 0x100004 /* Flash Control Register 1 */
#define FDR0_OFFSET 0x100008 /* Flash Data Register 0 */
#define FDR1_OFFSET 0x10000c /* Flash Data Register 1 */
#define FAR_OFFSET  0x100010 /* Flash Address Register */
#define FER_OFFSET  0x100014 /* Flash Error Register */
#define WPAR_OFFSET 0x10dfb0 /* Flash Non Volatile Write Protection Register */
#define APR0_OFFSET 0x10dfb8 /* Flash Non Volatile Access Protection Register 0 */
#define APR1_OFFSET 0x10dfbc /* Flash Non Volatile Access Protection Register 1 */

/* Flash Control Register 0 bitmasks */
#define FCR0_WMS   0x80000000
#define FCR0_SUSP  0x40000000
#define FCR0_WPG   0x20000000
#define FCR0_DWPG  0x10000000
#define FCR0_SER   0x8000000
#define FCR0_BER   0x4000000
#define FCR0_MER   0x2000000
#define FCR0_SPR   0x1000000
#define FCR0_INTM  0x200000
#define FCR0_INTP  0x100000
#define FCR0_LPS   0x8000
#define FCR0_LOCK  0x10
#define FCR0_BUSY1 0x4
#define FCR0_BUSY0 0x2

/* Flash Error Register bitmasks */
#define FER_WPF   0x100
#define FER_RESER 0x80
#define FER_SEQER 0x40
#define FER_IOER  0x8
#define FER_PGER  0x4
#define FER_ERER  0x2
#define FER_ERR   0x1
#define FER_MASK  0xffffffff

static libmem_geometry_t geometry[] =
{
  { 4, 0x00002000 },
  { 1, 0x00008000 },
  { 3, 0x00010000 },
  { 0, 0 }
};

static inline int
flash_wait_while_busy(unsigned long timeout_ticks)
{
  unsigned long start_ticks = libmem_get_ticks();
  do
    {
      if (!FLASH_REG(FCR0_OFFSET))
        {
          unsigned long fer = FLASH_REG(FER_OFFSET);
          if (fer)
            {
              FLASH_REG(FER_OFFSET) = 0;
              if (fer & (1 << 8))
                return LIBMEM_STATUS_LOCKED;
              return LIBMEM_STATUS_ERROR;
            }
          return LIBMEM_STATUS_SUCCESS;
        }
    }
  while (timeout_ticks == 0 || (libmem_get_ticks() - start_ticks) < timeout_ticks);
  return LIBMEM_STATUS_TIMEOUT;
}

static int
flash_program(libmem_driver_handle_t *h, flash_pointer_t dest, flash_pointer_t src)
{
  /* send flash write command to the flash */
  FLASH_REG(FCR0_OFFSET) |= FCR0_WPG;
  FLASH_REG(FAR_OFFSET) = (unsigned long)dest;
  FLASH_REG(FDR0_OFFSET) = *src;
  FLASH_REG(FCR0_OFFSET) |= FCR0_WMS;
  return flash_wait_while_busy(h->flash_info ? h->flash_info->write_timeout_ticks : 0);
}

static unsigned long erase_sector_mask;

static int
flash_erase_sector(libmem_driver_handle_t *h, libmem_sector_info_t *si)
{
  erase_sector_mask |= 1 << si->number;
  return LIBMEM_STATUS_SUCCESS;
}

static int
libmem_write_impl(libmem_driver_handle_t *h, unsigned char *dest, const unsigned char *src, size_t size)
{
  int res;
  if (size)
    {
      unsigned char *aligned_dest = LIBMEM_ALIGNED_ADDRESS(dest, 4);
      if (aligned_dest != dest)
        {
          unsigned char tmp[4];
          unsigned long offs = dest - aligned_dest;
          size_t frag = 4 - offs;
          if (frag > size)
            frag = size;
          memcpy(tmp, aligned_dest, 4);
          memcpy(tmp + offs, src, frag);
          res = flash_program(h, (flash_pointer_t)aligned_dest, (flash_pointer_t)tmp);
          if (res != LIBMEM_STATUS_SUCCESS)
            return res;
          src += frag;
          dest += frag;
          size -= frag;
        }
      if (LIBMEM_ADDRESS_IS_ALIGNED(src, 4))
        {
          while (size >= 4)
            {
              res = flash_program(h, (flash_pointer_t)dest, (flash_pointer_t)src);
              if (res != LIBMEM_STATUS_SUCCESS)
                return res;
              src += 4;
              dest += 4;
              size -= 4;
            }
        }
      else
        {
          while (size >= 4)
            {
              unsigned char tmp[4];
              memcpy(tmp, src, 4);
              res = flash_program(h, (flash_pointer_t)dest, (flash_pointer_t)tmp);
              if (res != LIBMEM_STATUS_SUCCESS)
                return res;
              src += 4;
              dest += 4;
              size -= 4;
            }
        }
      if (size)
        {
          unsigned char tmp[4];
          memcpy(tmp, dest, 4);
          memcpy(tmp, src, size);
          res = flash_program(h, (flash_pointer_t)dest, (flash_pointer_t)tmp);
          if (res != LIBMEM_STATUS_SUCCESS)
            return res;
        }
    }
  return LIBMEM_STATUS_SUCCESS;
}

static int
libmem_fill_impl(libmem_driver_handle_t *h, unsigned char *dest, unsigned char c, size_t size)
{
  int res;
  unsigned char cccc[4] = { (unsigned char)c, (unsigned char)c, (unsigned char)c, (unsigned char)c };
  if (size)
    {
      unsigned char *aligned_dest = LIBMEM_ALIGNED_ADDRESS(dest, 4);
      if (aligned_dest != dest)
        {
          unsigned char tmp[4];
          unsigned long offs = dest - aligned_dest;
          size_t frag = 4 - offs;
          if (frag > size)
            frag = size;
          memcpy(tmp, aligned_dest, 4);
          memset(tmp + offs, c, frag);
          res = flash_program(h, (flash_pointer_t)aligned_dest, (flash_pointer_t)tmp);
          if (res != LIBMEM_STATUS_SUCCESS)
            return res;
          dest += frag;
          size -= frag;
        }
      while (size >= 4)
        {
          res = flash_program(h, (flash_pointer_t)dest, (flash_pointer_t)&cccc);
          if (res != LIBMEM_STATUS_SUCCESS)
            return res;
          dest += 4;
          size -= 4;
        }
      if (size)
        {
          unsigned char tmp[4];
          memcpy(tmp, dest, 4);
          memset(tmp, c, size);
          res = flash_program(h, (flash_pointer_t)dest, (flash_pointer_t)tmp);
          if (res != LIBMEM_STATUS_SUCCESS)
            return res;
        }
    }
  return LIBMEM_STATUS_SUCCESS;
}

static int
libmem_erase_impl(libmem_driver_handle_t *h, unsigned char *dest, size_t size, unsigned char **erased_start, size_t *erased_size)
{ 
  int res = LIBMEM_STATUS_SUCCESS;
  dest = (unsigned char *)((unsigned int)dest & 0xE01FFFFF); /* Mask address in case one of the alias locations is being used */
  erase_sector_mask = 0;
  libmem_foreach_sector_in_range(h, dest, size, flash_erase_sector, erased_start, erased_size);
  if (erase_sector_mask)
    {
      FLASH_REG(FCR0_OFFSET) |= FCR0_SER;
      FLASH_REG(FCR1_OFFSET) |= erase_sector_mask;
      FLASH_REG(FCR0_OFFSET) |= FCR0_WMS;
      res = flash_wait_while_busy(h->flash_info ? h->flash_info->erase_sector_timeout_ticks : 0);
    }
  return res;
}

static int
libmem_flush_impl(libmem_driver_handle_t *h)
{
  return LIBMEM_STATUS_SUCCESS;
}

static const libmem_driver_functions_t driver_functions =
{
  libmem_write_impl,
  libmem_fill_impl,
  libmem_erase_impl,
  0,
  0,
  libmem_flush_impl
};

int
libstr73x_register_libmem_driver(libmem_driver_handle_t *h)
{
  libmem_register_driver(h, (unsigned char *)FLASH_START_ADDRESS, 0x00040000, geometry, 0, &driver_functions, 0);
  return LIBMEM_STATUS_SUCCESS;
}
