// CrossWorks Tasking Library.
//
// Copyright (c) 2005 Rowley Associates Limited.
//
// This file may be distributed under the terms of the License Agreement
// provided with this software.
//
// THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#include "ctl_api.h"
#include <targets/STR73x.h>

static CTL_ISR_FN_t isrtable[64];

void irq_handler(void) __attribute__((naked));

#define SIR(n) (*(&EIC_SIR0 + n))

void
irq_handler(void)
{
  // check if I bit is set in SPSR to detect spurious interrupts
  asm("stmfd sp!, {r0}");
  asm("mrs r0, spsr");
  asm("tst r0, #0x80");
  asm("ldmfd sp!, {r0}");
  asm("subnes pc, lr, #4");
#ifdef CTL_TASKING
  // store all registers in the tasking case
  asm("stmfd sp!, {r0-r12, lr}"); 
  asm("mrs r0, spsr");
  asm("stmfd sp!, {r0}");
  ctl_interrupt_count++;
#else
  // store the APCS registers in the non-tasking case
  asm("stmfd sp!, {r0-r4, r12, lr}");
#endif
  ((CTL_ISR_FN_t)isrtable[EIC_IVR])();
  /* Signal EOI */
  if (EIC_CICR < 32)
    EIC_IPR0 = 1 << EIC_CICR;
  else
    EIC_IPR1 = (1 << (EIC_CICR-32));
#ifdef CTL_TASKING
  asm("mov r0, sp");
  asm("ldr r1, =ctl_exit_isr");
  asm("bx r1");
#else
  // return from interrupt
  asm("ldmfd sp!, {r0-r4, r12, lr}");
  asm("subs pc, lr, #4");
#endif
}

int
ctl_set_isr(unsigned int vector, unsigned int priority, CTL_ISR_TRIGGER_t trigger, CTL_ISR_FN_t isr, CTL_ISR_FN_t *oldisr)
{
  if (oldisr)
    *oldisr = (CTL_ISR_FN_t)(isrtable[(SIR(vector) >> 16)]);
  isrtable[vector] = isr;
  SIR(vector) = (vector << 16) | (priority & 0xF);
  EIC_IVR = 0;
  return 1;
}

int
ctl_unmask_isr(unsigned int vector)
{
  if (vector < 32)
    EIC_IER0 |= 1 << vector; /* Enable interrupt vector */
  else
    EIC_IER1 |= (1 << (vector-32));
  EIC_ICR |= 1; /* Enable global interrupts */
  return 1;
}

int
ctl_mask_isr(unsigned int vector)
{
  if (vector < 32)
    EIC_IER0 &= ~(1 << vector);
  else
    EIC_IER1 &= ~(1 << (vector-32));
  return 1;
}

static CTL_ISR_FN_t userTimerISR;

static void
timerISR(void)
{
  unsigned int x;
  userTimerISR();  
  TB0_SR = 0; // Clear timer interrupt
}

void
ctl_start_timer(CTL_ISR_FN_t isr)
{
  CFG_PCGR0 |= CFG_PCGR0_EIC_MASK | CFG_PCGR0_TB0_MASK; // enable IEC and TB0
  int i;
  for (i=0;i<100;i++); // wait a bit for TB0 to wakeup
  TB0_VR = 99; // Set the down counter
  TB0_PR = 237; // Set the prescaler
  TB0_MR |= TB0_MR_ECM; // Generate interrupt on end count
  TB0_CR |= TB0_CR_SC; // Turn on the timer

  userTimerISR = isr;
  ctl_set_isr(21, 2, CTL_ISR_TRIGGER_FIXED, timerISR, 0);
  ctl_unmask_isr(21);
}

unsigned long
ctl_get_ticks_per_second()
{
  return 1000;
}


