// CrossWorks Tasking Library.
//
// Copyright (c) 2004 Rowley Associates Limited.
//
// This file may be distributed under the terms of the License Agreement
// provided with this software.
//
// THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#include <targets/MC9328MXL.h>
#include "ctl_api.h"

#define INTERRUPT_SOURCE_COUNT 64

static CTL_ISR_FN_t vectors[INTERRUPT_SOURCE_COUNT];

#define AITC_NIPRIORITY(n) (*(&AITC_NIPRIORITY7 + ((63 - n) / 8)))

void irq_handler(void) __attribute__((naked));

static void
irq_handler2(void)
{
  unsigned long vecsr = AITC_NIVECSR;
  /* Store current interrupt mask */
  unsigned long nimask = AITC_NIMASK; 
  /* Set the interrupt mask to the current interrupt priority level in order
     to allow only higher priority interrupts to interrupt the ISR if interrupts
     are re-enabled */
  AITC_NIMASK = vecsr & 0x1F;
  /* Call ISR */
  vectors[vecsr >> 16]();
  /* Restore interrupt mask */
  AITC_NIMASK = nimask; 
}

void
irq_handler(void)
{
  // check if I bit is set in SPSR to detect spurious interrupts
  asm("stmfd sp!, {r0}");
  asm("mrs r0, spsr");
  asm("tst r0, #0x80");
  asm("ldmfd sp!, {r0}");
  asm("subnes pc, lr, #4");
#ifdef CTL_TASKING
  asm("stmfd sp!, {r0-r12, lr}"); 
  asm("mrs r0, spsr");
  asm("stmfd sp!, {r0}");
  /* Increment interrupt count */
  ctl_interrupt_count++;
#else
  // store the APCS registers in the non-tasking case
  asm("stmfd sp!, {r0-r4, r12, lr}");
#endif
  irq_handler2();
#ifdef CTL_TASKING
  asm("mov r0, sp");
  asm("ldr r1, =ctl_exit_isr");
  asm("bx r1");
#else
  // return from interrupt
  asm("ldmfd sp!, {r0-r4, r12, lr}");
  asm("subs pc, lr, #4");
#endif
}

int
ctl_set_isr(unsigned int vector, unsigned int priority, CTL_ISR_TRIGGER_t trigger, CTL_ISR_FN_t isr, CTL_ISR_FN_t *oldisr)
{
  if (oldisr)
    *oldisr = vectors[vector];
  vectors[vector] = isr;
  /* Make sure interrupt is an IRQ interrupt */
  if (vector >= 32)
    AITC_INTTYPEH &= ~(1 << (vector - 32));
  else
    AITC_INTTYPEL &= ~(1 << vector);
  /* Set the interrupt priority level */
  AITC_NIPRIORITY(vector) &= ~(0xF << ((vector % 8) * 4));
  AITC_NIPRIORITY(vector) |= priority << ((vector % 8) * 4);
  return 1;
}

int
ctl_unmask_isr(unsigned int vector)
{
  if (vector >= 32)
    AITC_INTENABLEH |= 1 << (vector - 32);
  else
    AITC_INTENABLEL |= 1 << vector;
  return 1;
}

int
ctl_mask_isr(unsigned int vector)
{
  if (vector >= 32)
    AITC_INTENABLEH &= ~(1 << (vector - 32));
  else
    AITC_INTENABLEL |= ~(1 << vector);
  return 1;
}

#define PERCLK1_FREQUENCY 96000000

#define TIMER1_INT 59

static CTL_ISR_FN_t userTimerISR;

static void
timerISR(void)
{
  userTimerISR();
  /* Clear the timer interrupt */
  TIMER1_TSTAT &= ~1;
}

void
ctl_start_timer(CTL_ISR_FN_t isr)
{
  /* Reset timer 1 */
  TIMER1_TCTL |= 0x00000001;
  TIMER1_TCTL |= 0x00008000;
  TIMER1_TCTL &= ~0x00000001;

  /* Configure and enable timer 1 */
  TIMER1_TPRER = 0; /* Prescaler = 1 */
  TIMER1_TCMP = PERCLK1_FREQUENCY / 1000; /* Generate 1Khz timer interrupt */
  TIMER1_TCTL = 0x00000013; /* Enable timer, clock source = PERCLK1, enable interrupt, restart mode */

  userTimerISR = isr;
  ctl_set_isr(TIMER1_INT, 1, CTL_ISR_TRIGGER_LOW_LEVEL, timerISR, 0);
  ctl_unmask_isr(TIMER1_INT);
}

unsigned long
ctl_get_ticks_per_second(void)
{
  return 1000;
}