// CrossWorks Tasking Library.
//
// Copyright (c) 2004 Rowley Associates Limited.
//
// This file may be distributed under the terms of the License Agreement
// provided with this software.
//
// THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

#include <targets/ADuC7020.h>
#include "ctl_api.h"

#define INTERRUPT_SOURCE_COUNT 24

static CTL_ISR_FN_t vectors[INTERRUPT_SOURCE_COUNT];

static void
irq_handler2(void)
{
  unsigned int status = IRQSTA;
  int i;
  for (i = 1; i <= INTERRUPT_SOURCE_COUNT; ++i)
    {
      status >>= 1;
      if (status & 1)
        {
          IRQCLR = 1 << i;
          vectors[i - 1]();
          IRQEN = 1 << i;
          break;
        }
    }
}

void irq_handler(void) __attribute__((naked));

void
irq_handler(void)
{
  // check if I bit is set in SPSR to detect spurious interrupts
  asm("stmfd sp!, {r0}");
  asm("mrs r0, spsr");
  asm("tst r0, #0x80");
  asm("ldmfd sp!, {r0}");
  asm("subnes pc, lr, #4");
#ifdef CTL_TASKING
  asm("stmfd sp!, {r0-r12, lr}");
  asm("mrs r0, spsr");
  asm("stmfd sp!, {r0}");
  ctl_interrupt_count++;
#else
  // store the APCS registers in the non-tasking case
  asm("stmfd sp!, {r0-r4, r12, lr}");
#endif  
  irq_handler2();
#ifdef CTL_TASKING
  asm("mov r0, sp");
  asm("ldr r1, =ctl_exit_isr");
  asm("bx r1");
#else
  // return from interrupt
  asm("ldmfd sp!, {r0-r4, r12, lr}");
  asm("subs pc, lr, #4");
#endif
}

int
ctl_set_isr(unsigned int vector, unsigned int priority, CTL_ISR_TRIGGER_t trigger, CTL_ISR_FN_t isr, CTL_ISR_FN_t *oldisr)
{
  if (vector == 0 || vector > INTERRUPT_SOURCE_COUNT)
    return 0;
  if (oldisr)
    *oldisr = vectors[vector - 1];
  vectors[vector - 1] = isr;
  return 1;
}

int
ctl_unmask_isr(unsigned int vector)
{
  IRQEN = 1 << vector;
  return 1;
}

int
ctl_mask_isr(unsigned int vector)
{
  IRQCLR = 1 << vector;
  return 1;
}

#define TIMER0_INT 2

#define CORE_CLOCK_FREQUENCY 45088000
#define TIMER_CORE_CLOCK_DIVIDER 16
#define TIMER_FREQUENCY 1000
#define TIMER_RECHARGE (CORE_CLOCK_FREQUENCY / TIMER_CORE_CLOCK_DIVIDER / TIMER_FREQUENCY)

static CTL_ISR_FN_t userTimerISR;

static void
timerISR(void)
{
  /* Restart the timer */
  T0LD = TIMER_RECHARGE;
  /* Call timer ISR */
  userTimerISR();
  /* Clear the timer 0 interrupt */
  T0CLRI = 0;
}

void
ctl_start_timer(CTL_ISR_FN_t isr)
{
  T0LD = TIMER_RECHARGE;
  T0CON = 0xC4; /* Timer 0 enable, periodic, core clock / 16 */
  userTimerISR = isr;
  ctl_set_isr(TIMER0_INT, 0, CTL_ISR_TRIGGER_FIXED, timerISR, 0);
  ctl_unmask_isr(TIMER0_INT);
}

unsigned long
ctl_get_ticks_per_second()
{
  return 1000;
}