/*================================< prog.h >==================================*/

/*------------------------< Constant Definitions >----------------------------*/

// FLASH MEMORY ALLOCATION (2MB Flash Device)
// Flash Block 0 is reserved for Bootloader
// Flash Blocks 1 thru 7 are reserved for future configuration/file use
// Flash Blocks 8 thru 23 are used for storing main program binary image (1MB)
// Flash Blocks 24 thru 38 are uncommited

#define PROG_BOOTSPACE		0
#define PROG_BOOTSTARTADD 	0x000000   	// Start of bootloader space in Flash
#define PROG_BOOTENDADD   	0x001fff		// End of bootloader space (8KB) in Flash

#define PROG_MAINSPACE		1
#define PROG_MAINSIZE		0x100000
#define PROG_MAINSTARTADD	0x010000		// Start of main space in Flash
#define PROG_MAINENDADD		0x10ffef
#define PROG_NAMEADD			0x10fff0		// 8 digit program name (future)
#define PROG_CHECKSUMADD	0x10fffc		// 16 bit checksum of program space
#define PROG_STATUSADD		0x10fffe		// 0x5555 indicates valid program

#define PROG_MAINVALID		0x5555		// Indicates valid main program

#define PROG_RAMSTART		0x20000000	// Start of SDRAM before remap
#define PROG_RAMEND			0x200fffff	// End of 1MB space

#define INTSRAM_START		0x60000000	// Start of Internal SRAM
#define INTSRAM_END			0x60001fff	// End of Internal SRAM bootloader

// Unit Errors
#define PROG_OK            0x00  // No Errors
#define PROG_NOT_ERASED    0x01  // Required Flash sectors not erased
#define PROG_ERASE_FAIL    0x02  // Flash Sector erase not successful
#define PROG_BAD_CSUM      0x03  // S19 checksum error
#define PROG_WRITE_FAIL    0x04  // Flash write failure
#define PROG_VERIFY_FAIL   0x05  // Flash verify error
#define PROG_S19_ERROR     0x06  // Error decoding S19 data
#define PROG_ADDRESS_BAD   0x07  // Address was outside limits
#define PROG_ADDR_ALLIGN   0x08  // Can't write to non-alligned addresses
#define PROG_COMPLETE      0xff  // Write to Flash completed successfully


// Other Stuff
#define S19_ADDR_OFFSET 0x00000  // Offset for storing program in Flash

/*------------------------< Function Prototypes >-----------------------------*/

/*-----------------------------< ProgInit >-----------------------------------*/
// Initializes the Prog unit.
// Parameters	:	none
// Returns		; 	nothing
void ProgInit(void);

/*---------------------------< ProgMainValid >--------------------------------*/
// Checks to see if main program space is flagged as being valid.
// Parameters	:	none
// Returns		; 	Always TRUE
BYTE ProgMainValid(void);

/*--------------------------< ProgSeMainValid >-------------------------------*/
// Sets a WORD in Flash to indicate that main program space is valid.
// Parameters	:	none
// Returns		; 	nothing
void ProgSetMainValid(void);

/*-------------------------< ProgRequestDownload >----------------------------*/
// Checks Jumper JP3.  If the jumper is fitted returns TRUE.
// Parameters	:	none
// Returns		; 	TRUE if JP3 is present
BYTE ProgRequestDownload(void);

/*---------------------------< ProgLoadIntoRAM >------------------------------*/
// Reads an S19 (srec) file from the serial port and stores it in binary form
// in a buffer in SDRAM..
// Parameters	: 	bSpace	Destination space to load (PROG or BOOT)
// Returns :		BYTE		0 if OK, else error code (see above)
BYTE ProgLoadIntoRAM(BYTE bSpace);

/*----------------------------< ProgIntoFlash >-------------------------------*/
// Takes the buffer loaded with ProgLoadIntoRAM() and stores it is Flash memory
// as either the MAIN application image, or the BOOT code.  Flash must be
// erased and ready for programming.
// Parameters	: 	bSpace	Destination space to load (PROG or BOOT)
// Returns :		BYTE		Always TRUE
BYTE ProgIntoFlash(BYTE bSpace);

/*----------------------------< ProgEraseFlash >------------------------------*/
// Erases either Boot or Main Application Space in Flash memory.
// Parameters	: 	bSpace	Flash space to erase (PROG or BOOT)
// Returns :		nothing
void ProgEraseFlash(BYTE bSpace);

/*----------------------------< ProgVerifyFlash >-----------------------------*/
// Compares specified space in Flash to the SDRAM buffer.  Sends report to
// terminal connected to Uart1.
// Parameters	: 	bSpace	Space to verify
// Returns :		BYTE		TRUE if contents match, else FALSE
BYTE ProgVerifyFlash(BYTE bSpace);

/*----------------------------< ProgCopyImage >-------------------------------*/
// Copies MAIN space contents from Flash to SDRAM for execution.  At this point
// SDRAM has not been remapped.
// Parameters	: 	none
// Returns :		nothing
void ProgCopyImage(void);

/*----------------------------< ProgExecuteMain >-----------------------------*/
// Remaps SDRAM to 0h and jumps there, hereby executing main application.
// Parameters	: 	none
// Returns :		never
void ProgExecuteMain(void);


// end of prog.h
