// Copyright (c) 2001-2005 Rowley Associates Limited.
//
// This file may be distributed under the terms of the License Agreement
// provided with this software.
//
// THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
//
////////////////////////////////////////////////////////////////////////////////
//
//          Logic Product Development SDKLH79524-10 UART Example
//
// Description
// -----------
// This example demonstrates configuring and writing to a UART. It also 
// demonstrates how to get printf output over the UART by implementing 
// __putchar.
//
// To see output:
//   - Connect serial cable from the UART connector on your target board to 
//     your host computer.
//   - Open CrossStudio's "Terminal Emulator Window". Configure it to 9600 baud, 
//     8 data bits, no parity, 1 stop bits. Click "Connect" to start the
//     terminal emulator window.
//
////////////////////////////////////////////////////////////////////////////////

#include <targets/LH79524.h>

#define UART_CLOCK_FREQUENCY 11289600

static void
delay(volatile unsigned long v)
{
  while (v--);
}

static void
UARTInitialize(unsigned int baud)
{
  unsigned int divisor = UART_CLOCK_FREQUENCY / (16 * baud);
  IOCON_MUXCTL5 = 0x0A00;
  RCPC_PCLKCTRL0 &= ~1; /* Enable clock to UART0 */
  UART0_IBRD = divisor;
  UART0_FBRD = 0;
  UART0_LCR_H = 0x60; /* 8 bit, 1 stop bit, no parity */
  UART0_CR = 0x301; /* Enable transmit and receive */
  delay(100);
}

static void
UARTWriteChar(unsigned char ch)
{
  while (UART0_FR & 0x20);
  UART0_DR = ch;
}

static unsigned char
UARTReadChar(void)
{
  while (UART0_FR & 0x10);
  return UART0_DR;
}

static int
UARTReadAvailable(void)
{
  return (UART0_FR & 0x10) == 0;
}

void 
__putchar(int ch)
{
  if (ch == '\n')
    UARTWriteChar('\r');
  UARTWriteChar(ch);
}

int
main(void)
{
  int i;
  UARTInitialize(9600);
  for (i = 0; ; ++i)
    {
      printf("Hello World (%d)\n", i);
      if (UARTReadAvailable())
        { 
          char ch = UARTReadChar();
          if (ch == 'q' || ch == 'Q')
            break;
          printf("Key \'%c\' pressed\n", ch);
        }
    }
  printf("Quit\n");
  return 0; 
}

