// Copyright (c) 2001-2004 Rowley Associates Limited.
//
// This file may be distributed under the terms of the License Agreement
// provided with this software.
//
// THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE
// WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
//
////////////////////////////////////////////////////////////////////////////////
//
//                    Aeroflex AX07CF192 - UART Example
//
// Description
// -----------
// This example demonstrates configuring and writing to a UART. It also 
// demonstrates how to get printf output over the UART by implementing 
// __putchar.
//
// To see output:
//   - Connect serial cable from the UART 0 connector on your target board to 
//     your host computer.
//   - Open CrossStudio's "Terminal Emulator Window". Configure it to 4800 baud, 
//     8 data bits, no parity, 1 stop bits. Click "Connect" to start the
//     terminal emulator window.
//
////////////////////////////////////////////////////////////////////////////////

#include <targets/AX07CF192.h>

#define PROCESSOR_CLOCK_FREQUENCY 33000000

static void
UARTInitialize(unsigned int baud)
{
  unsigned int divisor = PROCESSOR_CLOCK_FREQUENCY / (16 * baud);
  U0LCR = 0x03; /* 8 bit, 1 stop bit, no parity */

  U0LCR |= 0x80; /* Access divisor latch */
  U0DLL = divisor & 0xFF; /* Set divisor latch low byte */
  U0DLM = (divisor >> 8) & 0xFF; /* Set divisor latch high byte */
  U0LCR &= ~0x80; 
  UCLKDR = 1; /* Set UART Clock Divisor */
  UCLKCR = 1; /* Enable UART Clock */
}

static void
UARTWriteChar(unsigned char ch)
{
  while ((U0LSR & 0x40) == 0);
  U0THR = ch;
}

static unsigned char
UARTReadChar(void)
{
  while ((U0LSR & 0x01) == 0);
  return U0RBR;
}

static int
UARTReadAvailable(void)
{
  return U0LSR & 0x01;
}

void 
__putchar(int ch)
{
  if (ch == '\n')
    UARTWriteChar('\r');
  UARTWriteChar(ch);
}

int
main(void)
{
  int i;
  UARTInitialize(4800);
  for (i = 0; ; ++i)
    {
      printf("Hello World (%d)\n", i);
      if (UARTReadAvailable())
        { 
          char ch = UARTReadChar();
          if (ch == 'q' || ch == 'Q')
            break;
          printf("Key \'%c\' pressed\n", ch);
        }
    }
  printf("Quit\n");
  return 0; 
}

