/*****************************************************************************
 * Loader for Analog ADuC7020                                                *
 *                                                                           *
 * Copyright (c) 2004 Rowley Associates Limited.                             *
 *                                                                           *
 * This file may be distributed under the terms of the License Agreement     *
 * provided with this software.                                              *
 *                                                                           *
 * THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE   *
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. *
 *****************************************************************************/

#include "../loader/loader.h"
#include <targets/ADuC7020.h>

#define FLASH_START_ADDRESS 0x00080000
#define FLASH_SECTOR_SIZE 0x00000200
#define FLASH_SECTORS 124
#define FLASH_SIZE (FLASH_SECTOR_SIZE * FLASH_SECTORS)
#define FLASH_END_ADDRESS FLASH_START_ADDRESS + FLASH_SIZE

#define ADDRESS_IN_FLASH(a) ADDRESS_IN_RANGE((unsigned long)a, FLASH_START_ADDRESS, FLASH_END_ADDRESS)

#define FEECON_SINGLE_READ            0x01
#define FEECON_SINGLE_WRITE           0x02
#define FEECON_ERASE_WRITE            0x03
#define FEECON_SINGLE_VERIFY          0x04
#define FEECON_SINGLE_ERASE           0x05
#define FEECON_MASS_ERASE             0x06
#define FEECON_BURST_READ             0x07
#define FEECON_BURST_READ_WRITE       0x08
#define FEECON_ERASE_BURST_READ_WRITE 0x09
#define FEECON_BURST_TERMINATION      0x0A
#define FEECON_SIGNATURE              0x0B
#define FEECON_PROTECT                0x0C
#define FEECON_PING                   0x0F

#define FEESTA_COMMAND_COMPLETE       (1 << 0)
#define FEESTA_COMMAND_FAIL           (1 << 1)
#define FEESTA_CONTROLLER_BUSY        (1 << 2)
#define FEESTA_INTERRUPT_STATUS       (1 << 3)
#define FEESTA_BURST_COMMAND_ENABLE   (1 << 5)

static int
flash_write_uint16(volatile unsigned short *address, unsigned short data)
{
  while (FEESTA & FEESTA_CONTROLLER_BUSY);
  FEEADR = (unsigned int)address - FLASH_START_ADDRESS;
  FEEDAT = data;
  FEECON = FEECON_SINGLE_WRITE;
  return 1;
}

static int
flash_write_byte(volatile unsigned char *address, unsigned char data)
{
  int result = 1;
  unsigned char tmp[2];
  unsigned short *alignedAddress = (unsigned short *)((unsigned int)address & ~1);
  *(unsigned short *)tmp = *alignedAddress;
  tmp[(unsigned int)address & 1] = data;
  if (*(unsigned short *)tmp != *alignedAddress)
    result = flash_write_uint16((unsigned short *)alignedAddress, *(unsigned short *)tmp);
  return result;
}

static int
flash_erase_sector(volatile unsigned short *address)
{
  while (FEESTA & FEESTA_CONTROLLER_BUSY);
  FEEADR = (unsigned int)address - FLASH_START_ADDRESS;
  FEECON = FEECON_SINGLE_ERASE;
  return 1;
}

static int
flash_erase_all()
{
  while (FEESTA & FEESTA_CONTROLLER_BUSY);
  FEECON = FEECON_MASS_ERASE;
  return 1;
}

static int
is_erased(volatile unsigned int address, volatile unsigned int highAddress)
{
  unsigned int length = highAddress - address + 1;
  while (address & 3)
    {
      if (*(volatile unsigned char *)address != 0xFF)
        return 0;
      length--;
      address++;
    }
  while (length > 4)
    {
      if (*(volatile unsigned long *)address != 0xFFFFFFFF)
        return 0;
      length -= 4;
      address += 4;
    }
  while (length)
    {
      if (*(volatile unsigned char *)address != 0xFF)
        return 0;
      length--;
      address++;
    }
  return 1;
}

void
loaderBegin()
{
}

void
loaderEnd()
{
}

int
loaderPoke(unsigned char *address, unsigned int length)
{
  int res = 1;
  if (ADDRESS_IN_FLASH(address))
    {
      // Write aligned data
      if (((unsigned int)address & 1) == 0)
        {
          while (length >= 4)
            {
              unsigned int data = loaderReadWord();
              res &= flash_write_uint16((unsigned short *)address,  (unsigned short)data);
              address += 2;
              data >>= 16;
              flash_write_uint16((unsigned short *)address,  (unsigned short)data);
              address += 2;
              length -= 4;
            }
        }
    
      // Write unaligned data
      while (length)
        {
          unsigned int data = loaderReadWord();
          int i;
          for(i = 4; i && length; --i)
            {
              res &= flash_write_byte(address++, (unsigned char)data);
              data >>= 8;
              length--;
            }
        }
    }
  else
    {
      while(length)
        {
          unsigned int data = loaderReadWord();
          int i;
          for(i = 4; i && length; --i)
            {
              *address++ = (unsigned char)data;
              data >>= 8;
              length--;
            }
        }
    }
  return res;
}

int
loaderPeek(unsigned char *address, unsigned int length)
{
  unsigned int data;
  while(length)
    { 
      int i;             
      for(i = 4; i; --i)
        {
          data >>= 8;
          if(length)
            {
              data |= *address++ << 24;
              --length;
            }                      
        }
      loaderWriteWord(data);
    }
  return 1;
}

int
loaderMemset(unsigned char *address, unsigned int length,  unsigned char c)
{
  if (ADDRESS_IN_FLASH(address))
    {
      // Write aligned data
      unsigned short sc = c | (c << 8);
      if (((unsigned int)address & 1) == 0)
        {
          while (length >= 2)
            {
              if (!flash_write_uint16((unsigned short *)address,  sc))
                return 0;
              address += 2;
              length -= 2;
            }
        }
    
      // Write unaligned data
      while (length)
        {
          if (!flash_write_byte(address, c))
            return 0;
          ++address;
          --length;
        }
    }
  else
    {
      while(length--)
        *address++ = (unsigned char)c;
    }
  return 1;
}

int
loaderVerify(unsigned char *address, unsigned int length)
{
  int result = 1;
  while(length)
    {
      unsigned int data = loaderReadWord();
      int i;
      for(i = 4; i && length; --i)
        {
          if (*address++ != (unsigned char)data)
            result = 0;
          data >>= 8;
          length--;
        }
    }
  return result;
}

int
loaderErase(unsigned char *address, unsigned int length)
{
  unsigned int i;
  unsigned int highAddress = (unsigned int)address + length - 1;
  unsigned int sectorStartAddress = FLASH_START_ADDRESS;
  unsigned int sectorEndAddress = FLASH_START_ADDRESS + FLASH_SECTOR_SIZE - 1;
  for (i = 0; i < FLASH_SECTORS; ++i)
    {
      if (RANGE_OCCLUDES_RANGE((unsigned int)address, highAddress, sectorStartAddress, sectorEndAddress) &&
          !is_erased((unsigned int)address > sectorStartAddress ? (unsigned int)address : sectorStartAddress, highAddress < sectorEndAddress ? highAddress : sectorEndAddress))
        {
          /* Sector hasn't been erased and it needs to be, erase it. */
          if (!flash_erase_sector((unsigned short *)sectorStartAddress))
            return 0;
        }
      sectorStartAddress += FLASH_SECTOR_SIZE;
      sectorEndAddress += FLASH_SECTOR_SIZE;
    }
  return 1;
}

int
loaderEraseAll()
{
  return flash_erase_all((unsigned short *)FLASH_START_ADDRESS);
}

int
loaderSetParameter(unsigned int parameter, unsigned int value)
{
  return 1;
}

