/*****************************************************************************
 * Loader for Philips LPC210X                                                *
 *                                                                           *
 * Copyright (c) 2001, 2002, 2003 Rowley Associates Limited.                 *
 *                                                                           *
 * This file may be distributed under the terms of the License Agreement     *
 * provided with this software.                                              *
 *                                                                           *
 * THIS FILE IS PROVIDED AS IS WITH NO WARRANTY OF ANY KIND, INCLUDING THE   *
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. *
 *****************************************************************************/

#include "../loader/loader.h"

#define IAP_LOCATION 0x7FFFFFF1

#define IAP_CMD_PREPARE_SECTORS_FOR_WRITE_OPERATION 50
#define IAP_CMD_COPY_RAM_TO_FLASH 51
#define IAP_CMD_ERASE_SECTORS 52
#define IAP_CMD_BLANK_CHECK_SECTORS 53
#define IAP_CMD_READ_PART_ID 54
#define IAP_CMD_READ_BOOT_CODE_VERSION 55
#define IAP_CMD_COMPARE 56

#define CMD_SUCCESS 0
#define INVALID_COMMAND 1
#define SRC_ADDR_ERROR 2
#define DST_ADDR_ERROR 3
#define SRC_ADDR_NOT_MAPPED 4
#define DST_ADDR_NOT_MAPPED 5
#define COUNT_ERROR 6
#define INVALID_SECTOR 7
#define SECTOR_NOT_BLANK 8
#define SECTOR_NOT_PREPARED_FOR_WRITE_OPERATION 9
#define COMPARE_ERROR 10
#define BUSY 11
#define PARAM_ERROR 12
#define ADDR_ERROR 13
#define ADDR_NOT_MAPPED 14
#define CMD_LOCKED 15
#define INVALID_CODE 16
#define INVALID_BAUD_RATE 17
#define INVALID_STOP_BIT 18

#define FLASH_SECTORS         15
#define FLASH_SECTOR_SIZE     0x2000
#define FLASH_START_ADDRESS   0x00000000
#define FLASH_END_ADDRESS     (FLASH_START_ADDRESS + (FLASH_SECTORS * FLASH_SECTOR_SIZE) - 1)
#define FLASH_INVALID_ADDRESS (FLASH_START_ADDRESS - 1)

#include "__armlib.h"
#include "targets/LPC210x.h"

typedef volatile unsigned int *reg32_t;

typedef void (*IAP)(unsigned long [], unsigned long[]);

static unsigned char currentSectorData[FLASH_SECTOR_SIZE];
static unsigned int currentSectorAddress = FLASH_INVALID_ADDRESS;
static unsigned int processorClockFrequency;

static unsigned long
iap_command(unsigned long cmd, unsigned long p0, unsigned long p1, unsigned long p2, unsigned long p3)
{
  static IAP iap_entry = (IAP)IAP_LOCATION;
  unsigned long command[5] = {cmd, p0, p1, p2, p3};
  unsigned long result[2];
  iap_entry(command, result);
  return result[0];
}

static int
is_erased(unsigned int address, unsigned int highAddress)
{
  unsigned int length = highAddress - address + 1;
  while (address & 3)
    {
      if (*(unsigned char *)address != 0xFF)
        return 0;
      length--;
      address++;
    }
  while (length > 4)
    {
      if (*(unsigned long *)address != 0xFFFFFFFF)
        return 0;
      length -= 4;
      address += 4;
    }
  while (length)
    {
      if (*(unsigned char *)address != 0xFF)
        return 0;
      length--;
      address++;
    }
  return 1;
}

static int
writeCurrentSector()
{
  // If current sector is in FLASH, write it.
  if (ADDRESS_IN_RANGE(currentSectorAddress, FLASH_START_ADDRESS, FLASH_END_ADDRESS))
    {
      int sectorNumber = (currentSectorAddress - FLASH_START_ADDRESS) / FLASH_SECTOR_SIZE;
      if (iap_command(IAP_CMD_PREPARE_SECTORS_FOR_WRITE_OPERATION, sectorNumber, sectorNumber, 0, 0) != CMD_SUCCESS)
        return 0;
      if (iap_command(IAP_CMD_COPY_RAM_TO_FLASH, currentSectorAddress, (unsigned long)currentSectorData, FLASH_SECTOR_SIZE, processorClockFrequency / 1000) != CMD_SUCCESS)
        return 0;
    }
  return 1;
}

static int
setOscillatorFrequency(unsigned int freq)
{
  processorClockFrequency = freq * (PLLCON & 1 ? (PLLCFG & 0xF) + 1 : 1);
}

void
loaderBegin()
{
  /* Set default value for oscillator frequency, this may overridden from host
     by setting a loader parameter. */
  setOscillatorFrequency(10000000);
}

void
loaderEnd()
{
}

int
loaderPoke(unsigned char *address, unsigned int length)
{
  while (length)
    {
      unsigned long data = loaderReadWord();
      int i;
      for(i = 4; i && length; --i)
        {
          unsigned int sectorAddress = (unsigned int)address / FLASH_SECTOR_SIZE * FLASH_SECTOR_SIZE;
          if (sectorAddress != currentSectorAddress)
            {
              writeCurrentSector();
              memcpy(currentSectorData, sectorAddress, FLASH_SECTOR_SIZE);
              if (!is_erased(sectorAddress, FLASH_SECTOR_SIZE))
                loaderErase((unsigned char *)sectorAddress, FLASH_SECTOR_SIZE);
              currentSectorAddress = sectorAddress;
            }
          currentSectorData[(unsigned int)address - currentSectorAddress] = (unsigned char)data;
          data >>= 8;
          --length;
          ++address;
        }
    }
  writeCurrentSector();
  currentSectorAddress = FLASH_INVALID_ADDRESS;
  return 1;
}

int
loaderPeek(unsigned char *address, unsigned int length)
{
  unsigned int data;
  while(length)
    { 
      int i;             
      for(i = 4; i; --i)
        {
          data >>= 8;
          if(length)
            {
              data |= *address++ << 24;
              --length;
            }                      
        }
      loaderWriteWord(data);
    }
  return 1;
}

int
loaderMemset(unsigned char *address, unsigned int length,  unsigned char c)
{
  while (length)
    {
      int offset, frag;
      unsigned int sectorAddress = (unsigned int)address / FLASH_SECTOR_SIZE * FLASH_SECTOR_SIZE;
      if (sectorAddress != currentSectorAddress)
        {
          writeCurrentSector();
          memset(currentSectorData, 0xFF, FLASH_SECTOR_SIZE);
          currentSectorAddress = sectorAddress;
        }
      offset = (unsigned int)address - currentSectorAddress;
      frag = FLASH_SECTOR_SIZE - offset;
      if (length < frag)
        frag = length;
      memset(currentSectorData + offset, c, frag);
      address += frag;
      length -= frag;
    }
  writeCurrentSector();
  currentSectorAddress = FLASH_INVALID_ADDRESS;
  return 1;
}

int
loaderVerify(unsigned char *address, unsigned int length)
{
  int result = 1;
  while(length)
    {
      unsigned int data = loaderReadWord();
      int i;
      for(i = 4; i && length; --i)
        {
          if (*address++ != (unsigned char)data)
            result = 0;
          data >>= 8;
          length--;
        }
    }
  return result;
}

int
loaderErase(unsigned char *address, unsigned int length)
{
  // Erase sectors. NOTE: This code assumes FLASH_START_ADDRESS
  // equals 0x00000000.
  int sectors, sectorStartNumber, sectorEndNumber;
  unsigned int sectorAddress;
  sectorAddress = (unsigned int)address / FLASH_SECTOR_SIZE * FLASH_SECTOR_SIZE;
  sectors = ((length - 1 + ((unsigned int)address - sectorAddress)) / FLASH_SECTOR_SIZE) + 1;
  sectorStartNumber = sectorAddress / FLASH_SECTOR_SIZE;
  if (sectorStartNumber < FLASH_SECTORS)
    {
      sectorEndNumber = sectorStartNumber + sectors - 1;
      if (sectorEndNumber >= FLASH_SECTORS)
        sectorEndNumber = FLASH_SECTORS - 1;
      iap_command(IAP_CMD_PREPARE_SECTORS_FOR_WRITE_OPERATION, sectorStartNumber, sectorEndNumber, 0, 0);
      iap_command(IAP_CMD_ERASE_SECTORS, sectorStartNumber, sectorEndNumber, processorClockFrequency / 1000, 0);
    }
  return 1;
}

int
loaderEraseAll()
{
  iap_command(IAP_CMD_PREPARE_SECTORS_FOR_WRITE_OPERATION, 0, FLASH_SECTORS - 1, 0, 0);
  iap_command(IAP_CMD_ERASE_SECTORS, 0, FLASH_SECTORS - 1, processorClockFrequency / 1000, 0);
  return 1;
}

int
loaderSetParameter(unsigned int parameter, unsigned int value)
{
  unsigned int result = 1;
  switch (parameter)
    {
      case 0:
        setOscillatorFrequency(value);
        break;
      default:
        break;
    }
  return result;
}

